<?php
/**
 * Base Repository
 *
 * @category GoMage
 * @package GoMage/Base
 * @author Stanislav Miroshnyk <smiroshnyk@gomage.com>
 * @copyright 2021 GoMage
 */
declare(strict_types=1);

namespace GoMage\Base\Model\Repository;

use Exception;
use GoMage\Base\Api\Data\EntityInterface;
use GoMage\Base\Api\Data\EntityInterfaceFactory;
use GoMage\Base\Api\RepositoryInterface;
use GoMage\Base\Helper\BaseRepository as BaseRepositoryHelper;
use GoMage\Base\Model\ResourceModel\Entity\CollectionFactory;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\Api\SearchResultsFactory;
use Magento\Framework\Api\SearchResultsInterface;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Model\ResourceModel\Db\AbstractDb as EntityResource;
use Magento\Framework\ObjectManager\Factory\AbstractFactory;
use Psr\Log\LoggerInterface;

/**
 * Class BaseRepository
 *
 * @package GoMage\Base\Model
 */
abstract class BaseRepository implements RepositoryInterface
{
    /**
     * @var AbstractFactory $entityFactory
     */
    protected $entityFactory;

    /**
     * @var AbstractFactory $entityCollectionFactory
     */
    protected $entityCollectionFactory;

    /**
     * OptionTitleRepository constructor.
     *
     * @param EntityResource $entityResource
     * @param CollectionProcessorInterface $collectionProcessor
     * @param SearchResultsFactory $searchResultsFactory
     * @param LoggerInterface $logger
     * @param BaseRepositoryHelper $baseRepositoryHelper
     * @param string $entityFactory
     * @param string $entityCollectionFactory
     */
    public function __construct(
        protected EntityResource $entityResource,
        protected CollectionProcessorInterface $collectionProcessor,
        protected SearchResultsFactory $searchResultsFactory,
        protected LoggerInterface $logger,
        protected BaseRepositoryHelper $baseRepositoryHelper,
        string $entityFactory = EntityInterfaceFactory::class,
        string $entityCollectionFactory = CollectionFactory::class
    ) {
        $objectManager = ObjectManager::getInstance();
        $this->entityFactory = $objectManager->create($entityFactory);
        $this->entityCollectionFactory = $objectManager->create($entityCollectionFactory);
    }

    /**
     * @inheritdoc
     */
    public function getNew(): EntityInterface
    {
        return $this->entityFactory->create();
    }

    /**
     * @inheritdoc
     */
    public function load($value, $field = null): EntityInterface
    {
        $item = $this->getNew();

        $this->entityResource->load($item, $value, $field);

        return $item;
    }

    /**
     * @inheritDoc
     */
    public function get(int $entityId): EntityInterface
    {
        $entity = $this->entityFactory->create();
        $this->entityResource->load($entity, $entityId);

        if (!$entity->getId()) {
            throw new NoSuchEntityException(__('Entity with id "%1" does not exist.', $entityId));
        }

        return $entity;
    }

    /**
     * @inheritDoc
     */
    public function getList(SearchCriteriaInterface $searchCriteria)
    {
        $collection = $this->entityCollectionFactory->create();
        $this->collectionProcessor->process($searchCriteria, $collection);

        /** @var SearchResultsInterface $searchResult */
        $searchResult = $this->searchResultsFactory->create();
        $searchResult->setSearchCriteria($searchCriteria);
        $searchResult->setTotalCount($collection->getSize());
        $searchResult->setItems($collection->getData());

        return $searchResult;
    }

    /**
     * @inheritDoc
     */
    public function getListByValue(string $columnName, string $value): array
    {
        $searchCriteria = $this->baseRepositoryHelper->getSearchCriteria($columnName, $value);
        $searchResult = $this->getList($searchCriteria);

        return $searchResult->getTotalCount() > 0 ? $searchResult->getItems() : [];
    }

    /**
     * @inheritDoc
     */
    public function getListByColumn(
        SearchCriteriaInterface $searchCriteria,
        string $columnName
    ) {
        $collection = $this->entityCollectionFactory->create();
        $this->collectionProcessor->process($searchCriteria, $collection);

        /** @var SearchResultsInterface $searchResult */
        $searchResult = $this->searchResultsFactory->create();
        $searchResult->setSearchCriteria($searchCriteria);
        $searchResult->setTotalCount($collection->getSize());
        $searchResult->setItems($collection->getColumnValues($columnName));

        return $searchResult;
    }

    /**
     * @inheritDoc
     */
    public function save(EntityInterface $entity): EntityInterface
    {
        $this->entityResource->save($entity);
        try {
        } catch (Exception $exception) {
            $this->logger->critical($exception->getMessage() . $exception->getTraceAsString());
            throw new CouldNotSaveException(__('Could not save the entity'), $exception);
        }

        return $entity;
    }

    /**
     * @inheritDoc
     */
    public function delete(EntityInterface $entity): bool
    {
        try {
            $this->entityResource->delete($entity);
        } catch (Exception $exception) {
            $this->logger->critical($exception->getMessage() . $exception->getTraceAsString());
            throw new CouldNotDeleteException(__('Could not delete the entity: %1', $exception->getMessage()));
        }

        return true;
    }

    /**
     * @inheritDoc
     */
    public function deleteById(int $entityId): bool
    {
        return $this->delete($this->get($entityId));
    }
}
