<?php
/**
 * Repository Interface
 *
 * @category GoMage
 * @package GoMage/Base
 * @author Stanislav Miroshnyk <smiroshnyk@gomage.com>
 * @copyright 2021 GoMage
 */
declare(strict_types=1);

namespace GoMage\Base\Api;

use GoMage\Base\Api\Data\EntityInterface;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\Api\SearchResults;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;

/**
 * Class RepositoryInterface
 *
 * @package GoMage\Base\Api
 */
interface RepositoryInterface
{
    /**
     * Get New Entity
     *
     * @return EntityInterface
     */
    public function getNew(): EntityInterface;

    /**
     * Load Entity by Field
     *
     * @param mixed $value
     *
     * @param null|mixed $field
     *
     * @return EntityInterface
     */
    public function load($value, $field = null): EntityInterface;

    /**
     * Save entity
     *
     * @param EntityInterface $entity
     *
     * @return EntityInterface
     *
     * @throws CouldNotSaveException
     */
    public function save(EntityInterface $entity): EntityInterface;

    /**
     * Get By Entity Id
     *
     * @param int $entityId
     *
     * @return EntityInterface
     *
     * @throws NoSuchEntityException
     */
    public function get(int $entityId): EntityInterface;

    /**
     * Get List
     *
     * @param SearchCriteriaInterface $searchCriteria
     *
     * @return SearchResults
     *
     * @throws LocalizedException
     */
    public function getList(
        SearchCriteriaInterface $searchCriteria
    );

    /**
     * Delete entity
     *
     * @param EntityInterface $entity
     *
     * @return bool
     *
     * @throws CouldNotDeleteException
     */
    public function delete(EntityInterface $entity): bool;

    /**
     * Delete entity by Id
     *
     * @param int $entityId
     *
     * @return bool
     *
     * @throws CouldNotDeleteException
     * @throws NoSuchEntityException
     */
    public function deleteById(int $entityId): bool;

    /**
     * Get List Of Column Values
     *
     * @param SearchCriteriaInterface $searchCriteria
     * @param string $columnName
     *
     * @return SearchResults
     *
     * @throws LocalizedException
     */
    public function getListByColumn(
        SearchCriteriaInterface $searchCriteria,
        string $columnName
    );

    /**
     * Get List By Value
     *
     * @param string $columnName
     * @param string $value
     *
     * @return array
     *
     * @throws LocalizedException
     */
    public function getListByValue(string $columnName, string $value): array;
}
