<?php

namespace GoAI\SmartBuy\Helper;

use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Framework\App\Helper\Context;
use Magento\Store\Model\ScopeInterface;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Checkout\Model\Cart;
use Magento\Quote\Model\QuoteIdToMaskedQuoteIdInterface;
use Magento\Quote\Model\QuoteIdMaskFactory;
use Magento\Framework\Math\Random;

class Data extends AbstractHelper
{
    private $storeManager;
    private $quoteIdToMaskedQuoteId;

    protected $quoteIdMaskFactory;
    private $cart;
    private $random;
    const CACHE_LIFETIME = 86400;
    const CACHE_NAME = 'goai_smart_buy_script_cache_';
    const XML_PATH_ENABLED = 'goai_smart_buy/general/enable';
    const XML_PATH_SCRIPT_URL = 'goai_smart_buy/general/script_url';


    /**
     * @param Context $context
     * @param StoreManagerInterface $storeManager
     * @param QuoteIdToMaskedQuoteIdInterface $quoteIdToMaskedQuoteId
     * @param Cart $cart
     * @param Random $random
     * @param QuoteIdMaskFactory $quoteIdMaskFactory
     */
    public function __construct(
        Context                         $context,
        StoreManagerInterface           $storeManager,
        QuoteIdToMaskedQuoteIdInterface $quoteIdToMaskedQuoteId,
        Cart                            $cart,
        Random                          $random,
        QuoteIdMaskFactory              $quoteIdMaskFactory

    )
    {
        parent::__construct($context);
        $this->storeManager = $storeManager;
        $this->quoteIdToMaskedQuoteId = $quoteIdToMaskedQuoteId;
        $this->cart = $cart;
        $this->random = $random;
        $this->quoteIdMaskFactory = $quoteIdMaskFactory;
    }

    /**
     * @return bool
     */
    public function isEnabled()
    {
        return $this->scopeConfig->isSetFlag(self::XML_PATH_ENABLED, ScopeInterface::SCOPE_STORE);
    }

    /**
     * @return mixed
     */
    public function getScriptUrl()
    {
        return $this->scopeConfig->getValue(self::XML_PATH_SCRIPT_URL, ScopeInterface::SCOPE_STORE);
    }

    /**
     * @return string
     * @throws \Magento\Framework\Exception\LocalizedException
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getOrCreateMaskedCartId()
    {
        // Get the current quote (cart)
        $quote = $this->cart->getQuote();

        // If the quote is not yet created, initialize and save it
        if (!$quote->getId()) {
            $quote->setStoreId($this->storeManager->getStore()->getId());
            $quote->save();
        }

        try {
            // Try to get the masked ID for the existing quote ID
            $maskedId = $this->quoteIdToMaskedQuoteId->execute($quote->getId());
            if (!$maskedId) {
                // If no masked ID is returned, create a new one
                $maskedId = $this->createMaskedIdForQuote($quote->getId());
            }
            return $maskedId;
        } catch (\Magento\Framework\Exception\NoSuchEntityException $e) {
            // If the masked ID does not exist, create and return a new one
            return $this->createMaskedIdForQuote($quote->getId());
        }
    }

    /**
     * @param $quoteId
     * @return string
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    protected function createMaskedIdForQuote($quoteId)
    {
        // Create a new instance of the masked quote model using the factory
        $quoteIdMask = $this->quoteIdMaskFactory->create();
        $quoteIdMask->setQuoteId($quoteId);

        // Generate a new masked ID (using random bytes)
        $maskedId = $this->random->getUniqueHash();
        $quoteIdMask->setMaskedId($maskedId);

        // Save the new masked ID record
        $quoteIdMask->save();
        return $maskedId;
    }

}
