<?php

namespace GoAI\SmartBuy\Controller\Cart;

use Magento\Framework\App\Action\Action;
use Magento\Framework\App\Action\Context;
use Magento\Framework\Controller\Result\RedirectFactory;
use Magento\Quote\Model\QuoteIdMaskFactory;
use Magento\Quote\Api\CartRepositoryInterface;
use Magento\Checkout\Model\Session as CheckoutSession;
use Magento\Framework\Exception\NoSuchEntityException;

class View extends Action
{
    /**
     * @var RedirectFactory
     */
    protected $resultRedirectFactory;

    /**
     * @var QuoteIdMaskFactory
     */
    protected $quoteIdMaskFactory;

    /**
     * @var CartRepositoryInterface
     */
    protected $cartRepository;

    /**
     * @var CheckoutSession
     */
    protected $checkoutSession;

    /**
     * Constructor
     *
     * @param Context $context
     * @param RedirectFactory $resultRedirectFactory
     * @param QuoteIdMaskFactory $quoteIdMaskFactory
     * @param CartRepositoryInterface $cartRepository
     * @param CheckoutSession $checkoutSession
     */
    public function __construct(
        Context $context,
        RedirectFactory $resultRedirectFactory,
        QuoteIdMaskFactory $quoteIdMaskFactory,
        CartRepositoryInterface $cartRepository,
        CheckoutSession $checkoutSession
    ) {
        $this->resultRedirectFactory = $resultRedirectFactory;
        $this->quoteIdMaskFactory = $quoteIdMaskFactory;
        $this->cartRepository = $cartRepository;
        $this->checkoutSession = $checkoutSession;
        parent::__construct($context);
    }

    /**
     * Execute action to load the cart based on the masked cart ID.
     * If the cart exists, it is associated with the user's session so that all items are displayed.
     *
     * @return \Magento\Framework\Controller\Result\Redirect
     */
    public function execute()
    {
        // Retrieve the masked cart ID from the URL parameter
        $maskedId = $this->getRequest()->getParam('masked_id');
        if (!$maskedId) {
            // If no masked ID is provided, add an error message and redirect to homepage
            $this->messageManager->addErrorMessage(__('Masked Cart ID is not specified.'));
            return $this->resultRedirectFactory->create()->setPath('/');
        }

        try {
            // Load the quote ID mask record by masked_id
            $quoteIdMask = $this->quoteIdMaskFactory->create()->load($maskedId, 'masked_id');
            if (!$quoteIdMask->getId()) {
                throw new NoSuchEntityException(__('Cart with the specified masked ID was not found.'));
            }

            // Retrieve the real quote ID from the masked record
            $quoteId = $quoteIdMask->getQuoteId();

            // Load the cart (quote) using the real quote ID
            $quote = $this->cartRepository->get($quoteId);

            // Associate the retrieved cart with the user's session
            $this->checkoutSession->replaceQuote($quote);

            // Redirect the user to the cart page to view all items
            return $this->resultRedirectFactory->create()->setPath('checkout/cart');

        } catch (\Exception $e) {
            // If an error occurs, display the error message and redirect to the homepage
            $this->messageManager->addErrorMessage($e->getMessage());
            return $this->resultRedirectFactory->create()->setPath('/');
        }
    }
}
