<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\Translator\Model;

use Exception;
use GoAi\Translator\Api\Data\TranslationInterface;
use GoAi\Translator\Api\Data\TranslationInterfaceFactory;
use GoAi\Translator\Api\Data\TranslationStoreInterface;
use GoAi\Translator\Api\TranslationQueueManagerInterface;
use GoAi\Translator\Api\TranslationRepositoryInterface;
use GoAi\Translator\Helper\Config as ConfigHelper;
use GoAi\Translator\Model\ResourceModel\Translation as TranslationResource;
use GoAi\Translator\Model\Config\Source\TranslationStatus;
use GoAi\Translator\Model\Config\Source\TranslationType;
use GoAi\Translator\Model\ResourceModel\Translation\CollectionFactory as TranslationCollectionFactory;
use GoAi\Translator\Model\ResourceModel\TranslationStore\Collection;
use GoAi\Translator\Model\ResourceModel\TranslationStore\CollectionFactory as TranslationStoreCollectionFactory;
use Magento\Framework\DB\Adapter\AdapterInterface;
use Magento\Framework\Exception\LocalizedException;
use Psr\Log\LoggerInterface;

/**
 * This class is responsible for managing translation queue.
 */
class TranslationQueueManager implements TranslationQueueManagerInterface
{
    /**
     * @var AdapterInterface|false
     */
    protected AdapterInterface|false $connection = false;

    /**
     * TranslationManager constructor.
     *
     * @param ConfigHelper $configHelper
     * @param EntityTranslatorFactory $entityTranslatorFactory
     * @param TranslationCollectionFactory $translationCollectionFactory
     * @param TranslationInterfaceFactory $translationFactory
     * @param TranslationRepositoryInterface $translationRepository
     * @param TranslationResource $translationResource
     * @param TranslationStoreCollectionFactory $translationStoreCollectionFactory
     * @param LoggerInterface $logger
     */
    public function __construct(
        protected ConfigHelper $configHelper,
        protected EntityTranslatorFactory $entityTranslatorFactory,
        protected TranslationCollectionFactory $translationCollectionFactory,
        protected TranslationInterfaceFactory $translationFactory,
        protected TranslationRepositoryInterface $translationRepository,
        protected TranslationResource $translationResource,
        protected TranslationStoreCollectionFactory $translationStoreCollectionFactory,
        protected LoggerInterface $logger
    ) {
    }

    /**
     * @inheritDoc
     */
    public function startOnboarding(int $storeId): void
    {
        $entityTypes = $this->configHelper->getEnabledEntityTypes($storeId);

        foreach ($entityTypes as $entityType) {
            $this->massAddToTranslationQueue($entityType, $storeId);
        }
    }

    /**
     * @inheritDoc
     */
    public function massAddToTranslationQueue(
        string $entityType,
        int $storeId,
        array $entityIds = [],
        int $translationType = TranslationType::SITE_ONBOARDING
    ): void {
        if ($entityIds === []) {
            $entityTranslator = $this->entityTranslatorFactory->create($entityType);
            $entityIds = $entityTranslator->getEntityIdsForTranslation();
        }

        if ($entityIds === []) {
            return;
        }

        //Exclude values that are already in the queue
        $entityIds = array_values(
            array_diff(
                $entityIds,
                array_unique(
                    $this->translationCollectionFactory->create()
                        ->addFieldToFilter(
                            TranslationInterface::ORIGINAL_ENTITY_ID,
                            ['in' => $entityIds]
                        )->addFieldToFilter(
                            TranslationInterface::ENTITY_TYPE,
                            $entityType
                        )->addFieldToFilter(
                            TranslationInterface::STATUS,
                            ['in' => [
                                TranslationStatus::STATUS_NEW,
                                TranslationStatus::STATUS_IN_PROGRESS,
                                TranslationStatus::STATUS_TRANSLATED,
                                TranslationStatus::STATUS_SKIPPED,
                                TranslationStatus::STATUS_PENDING,
                                TranslationStatus::STATUS_ERROR_ON_TRANSLATION,
                                TranslationStatus::STATUS_ERROR_ON_SAVING,
                                TranslationStatus::STATUS_ERROR_ON_PROCESSING
                            ]]
                        )->addStoreFilter(
                            $storeId
                        )->getColumnValues(TranslationInterface::ORIGINAL_ENTITY_ID)
                )
            )
        );

        if ($entityIds === []) {
            return;
        }

        $this->massInsertTranslations($entityIds, $entityType, $translationType);
        $this->assignTranslationsToStore(
            $this->translationCollectionFactory->create()
                ->addFieldToFilter(
                    TranslationInterface::ORIGINAL_ENTITY_ID,
                    ['in' => $entityIds]
                )->addFieldToFilter(
                    TranslationInterface::ENTITY_TYPE,
                    $entityType
                )->addFieldToFilter(
                    TranslationInterface::STATUS,
                    TranslationStatus::STATUS_NEW
                )->addFieldToFilter(
                    TranslationInterface::TRANSLATION_TYPE,
                    $translationType
                )->addWithoutStoreFilter()->getAllIds(),
            $storeId
        );
    }

    /**
     * Insert translations for the given entity IDs and entity type.
     *
     * @param array $entityIds
     * @param string $entityType
     * @param int $translationType
     * @return void
     * @throws LocalizedException
     */
    protected function massInsertTranslations(
        array $entityIds,
        string $entityType,
        int $translationType = TranslationType::SITE_ONBOARDING
    ): void {
        $this->getConnection()->insertMultiple(
            $this->translationResource->getMainTable(),
            array_map(
                static fn ($entityId) => [
                    TranslationInterface::TRANSLATION_TYPE => $translationType,
                    TranslationInterface::STATUS => TranslationStatus::STATUS_NEW,
                    TranslationInterface::ENTITY_TYPE => $entityType,
                    TranslationInterface::ORIGINAL_ENTITY_ID => $entityId,
                    TranslationInterface::COMMENT => 'Entity added for translation'
                ],
                $entityIds
            )
        );
    }

    /**
     * @return AdapterInterface|bool
     */
    protected function getConnection(): AdapterInterface|bool
    {
        if ($this->connection) {
            return $this->connection;
        }

        $this->connection = $this->translationResource->getConnection();

        return $this->connection;
    }

    /**
     * Assign translations to the specified store.
     *
     * @param array $translationIds
     * @param int $storeId
     * @return void
     */
    protected function assignTranslationsToStore(array $translationIds, int $storeId): void
    {
        $this->getConnection()->insertMultiple(
            $this->translationResource->getTable(TranslationStoreInterface::TABLE_NAME),
            array_map(
                static fn ($translationId) => [
                    TranslationStoreInterface::TRANSLATION_ID => $translationId,
                    TranslationStoreInterface::STORE_ID => $storeId,
                ],
                $translationIds
            )
        );
    }

    /**
     * @inheritDoc
     */
    public function addToTranslationQueue(
        string $entityType,
        int $entityId,
        array $storeIds,
        int $translationType = TranslationType::CONTENT_UPDATE
    ): void {
        if (!$this->configHelper->isTranslatorEnabled() || empty($storeIds)) {
            return;
        }

        try {
            $translationCollection = $this->translationCollectionFactory->create();
            $translationCollection->addFieldToFilter(
                TranslationInterface::ORIGINAL_ENTITY_ID,
                $entityId
            )->addFieldToFilter(
                TranslationInterface::ENTITY_TYPE,
                $entityType
            )->addFieldToFilter(
                TranslationInterface::STATUS,
                ['in' => [
                    TranslationStatus::STATUS_NEW,
                    TranslationStatus::STATUS_ERROR_ON_TRANSLATION,
                    TranslationStatus::STATUS_ERROR_ON_SAVING,
                    TranslationStatus::STATUS_ERROR_ON_PROCESSING
                ]]
            );

            $translation = $this->translationFactory->create();

            if ($translationCollection->getSize()) {
                $translation = $translationCollection->getFirstItem();

                $translationCollection->addFieldToFilter(
                    TranslationInterface::TRANSLATION_TYPE,
                    TranslationType::CONTENT_UPDATE
                )->clear();

                if ($translationCollection->getSize()) {
                    return;
                }
            }

            /** @var TranslationInterface $translation */
            $translation->setOriginalEntityId((int)$entityId)
                ->setEntityType($entityType)
                ->setTranslationType($translationType)
                ->setStatus(TranslationStatus::STATUS_NEW)
                ->setStoreIds($storeIds)
                ->setComment('Entity added for translation');
            $this->translationRepository->save($translation);
        } catch (Exception $exception) {
            $this->logger->critical(
                'Error while adding translation to queue: ' . $exception->getMessage(),
                ['exception' => $exception]
            );
        }
    }

    /**
     * @inheritDoc
     */
    public function updateTranslationStatus(
        int $translationId,
        int $status,
        string|null $message = null
    ): void {
        $this->massUpdateTranslationStatus([$translationId], $status, $message);
    }

    /**
     * @inheritDoc
     */
    public function massUpdateTranslationStatus(array $translationIds, int $status, string|null $message = null): void
    {
        if (!$this->configHelper->isTranslatorEnabled()) {
            return;
        }

        $connection = $this->translationResource->getConnection();
        $tableName = $this->translationResource->getMainTable();

        $updatedData = [
            TranslationInterface::STATUS => $status,
        ];

        if ($message) {
            $updatedData[TranslationInterface::COMMENT] = $message;
        }

        $connection->update(
            $tableName,
            $updatedData,
            [TranslationInterface::TRANSLATION_ID . ' IN (?)' => $translationIds]
        );
    }

    /**
     * @inheritDoc
     */
    public function getTotalEntitiesCount(int $storeId): int
    {
        if (!$this->configHelper->isTranslatorEnabled()) {
            return 0;
        }

        $entityTypes = $this->configHelper->getEnabledEntityTypes($storeId);
        $totalCount = 0;

        foreach ($entityTypes as $entityType) {
            $totalCount += $this->getTotalCountByEntityType($entityType);
        }

        return $totalCount;
    }

    /**
     * @inheritDoc
     */
    public function getTotalCountByEntityType(string $entityType): int
    {
        return count($this->entityTranslatorFactory->create($entityType)
            ->getEntityIdsForTranslation());
    }

    /**
     * @inheritDoc
     */
    public function getEntitiesCountInQueue(
        int $storeId,
        int $translationType = TranslationType::SITE_ONBOARDING
    ): int {
        if (!$this->configHelper->isTranslatorEnabled()) {
            return 0;
        }

        $collection = $this->translationCollectionFactory->create()
            ->addFieldToFilter(
                TranslationInterface::TRANSLATION_TYPE,
                $translationType
            )->addStoreFilter(
                $storeId
            )->addFieldToFilter(
                TranslationInterface::STATUS,
                ['neq' => TranslationStatus::STATUS_TRANSLATED]
            );

        return $collection->count();
    }

    /**
     * @inheritDoc
     */
    public function getEntitiesCountInQueueByEntityType(
        int $storeId,
        string $entityType,
        int $translationType = TranslationType::SITE_ONBOARDING
    ): int {
        if (!$this->configHelper->isTranslatorEnabled()) {
            return 0;
        }

        $collection = $this->translationCollectionFactory->create()
            ->addFieldToFilter(
                TranslationInterface::TRANSLATION_TYPE,
                $translationType
            )->addStoreFilter(
                $storeId
            )->addFieldToFilter(
                TranslationInterface::ENTITY_TYPE,
                $entityType
            )->addFieldToFilter(
                TranslationInterface::STATUS,
                ['neq' => TranslationStatus::STATUS_TRANSLATED]
            );

        return $collection->count();
    }

    /**
     * @inheritDoc
     */
    public function replaceDefaultStore(int $defaultStoreId): void
    {
        $connection = $this->getConnection();
        $tableName = $this->translationResource->getTable('goai_translation_store');
        $where = ['store_id IN (?)' => $defaultStoreId];

        $connection->delete($tableName, $where);

        /**
         * @var Collection $translationStoreCollection
         */
        $translationIds = array_unique($this->translationStoreCollectionFactory->create()
            ->getColumnValues(TranslationStoreInterface::TRANSLATION_ID));

        $connection = $this->getConnection();
        $tableName = $this->translationResource->getMainTable();
        $where = ['translation_id NOT IN (?)' => $translationIds];

        $connection->delete($tableName, $where);
    }
}
