<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\Translator\Model\Queue;

use Exception;
use GoAi\Translator\Api\TranslationQueueManagerInterface;
use GoAi\Translator\Helper\Config as ConfigHelper;
use GoAi\Translator\Helper\TermReplacement as TermReplacementHelper;
use GoAi\Translator\Model\Config\Source\TranslationStatus;
use GoAi\Translator\Model\EntityTranslatorFactory;
use GoAi\Translator\Model\ResourceModel\Translation\CollectionFactory as TranslationCollectionFactory;
use GoAi\Translator\Model\TranslationSender;
use Magento\Framework\Serialize\SerializerInterface;
use Psr\Log\LoggerInterface;

/**
 * TranslationHandler class
 */
class TranslationHandler
{
    /**
     * TranslationHandler constructor.
     *
     * @param ConfigHelper $configHelper
     * @param EntityTranslatorFactory $entityTranslatorFactory
     * @param SerializerInterface $serializer
     * @param TermReplacementHelper $termReplacementHelper
     * @param TranslationCollectionFactory $translationCollectionFactory
     * @param TranslationQueueManagerInterface $translationQueueManager
     * @param TranslationSender $translationSender
     * @param LoggerInterface $logger
     */
    public function __construct(
        protected ConfigHelper $configHelper,
        protected EntityTranslatorFactory $entityTranslatorFactory,
        protected SerializerInterface $serializer,
        protected TermReplacementHelper $termReplacementHelper,
        protected TranslationCollectionFactory $translationCollectionFactory,
        protected TranslationQueueManagerInterface $translationQueueManager,
        protected TranslationSender $translationSender,
        protected LoggerInterface $logger
    ) {
    }

    /**
     * Process translation
     * @param string $serializedData
     * @return void
     */
    public function process(string $serializedData): void
    {
        if (!$this->configHelper->isTranslatorEnabled()) {
            return;
        }

        $translationIds = [];

        try {
            $dataForTranslation = $this->serializer->unserialize($serializedData);

            if (!$this->isValidInputData($dataForTranslation)) {
                return;
            }

            $languages = $dataForTranslation['languages'] ?? [];
            $translationIds = $dataForTranslation['translation_ids'] ?? [];
            $translationIds = array_combine($translationIds, $translationIds);
            $translationCollection = $this->translationCollectionFactory->create()
                ->addFieldToFilter('translation_id', ['in' => $translationIds]);

            foreach ($translationCollection->getItems() as $translation) {
                $entityType = $translation->getEntityType();
                $entityTranslator = $this->entityTranslatorFactory->create($entityType);
                $translationId = (int)$translation->getId();
                $preparedData = array_merge(
                    ['translation_id' => $translationId],
                    $entityTranslator->getPreparedEntityData($translation->getOriginalEntityId())
                );
                uasort($preparedData, function ($a, $b) {
                    $lenA = is_scalar($a) ? strlen((string)$a) : -1;
                    $lenB = is_scalar($b) ? strlen((string)$b) : -1;
                    return $lenA <=> $lenB;
                });

                $storeIds = $translation->getStoreIds();
                $languageList = [];
                $languageMapping = [];

                foreach ($storeIds as $storeId) {
                    $language = $languages[$storeId] ?? null;

                    if ($language === null) {
                        $this->handleTranslationError(
                            $translationIds,
                            new Exception('Language not found for store ID: ' . $storeId)
                        );
                        return;
                    }

                    $languageList[$storeId] = $language;
                    $languageMapping[$language][] = (int)$storeId;
                }

                $this->processResult(
                    $this->translationSender->send(
                        $preparedData,
                        array_values(array_unique($languageList))
                    ),
                    $languageMapping,
                    $entityType,
                    $translationId
                );

                unset($translationIds[$translationId]);
            }
        } catch (Exception $exception) {
            $this->translationQueueManager->massUpdateTranslationStatus(
                $translationIds,
                TranslationStatus::STATUS_ERROR_ON_PROCESSING,
                'Error while preparing data for translation: ' . $exception->getMessage()
            );

            return;
        }
    }

    /**
     * Validate input data
     * @param array $data
     * @return bool
     */
    protected function isValidInputData(array $data): bool
    {
        if (empty($data['translation_ids'])) {
            $this->logger->error('Translation IDs are required');

            return false;
        }

        if (empty($data['languages'])) {
            $this->handleTranslationError(
                $data['translation_ids'],
                new Exception('Languages are required')
            );

            return false;
        }

        return true;
    }

    /**
     * Handle translation error
     * @param array $translationIds
     * @param Exception $exception
     * @return void
     */
    protected function handleTranslationError(array $translationIds, Exception $exception): void
    {
        $this->translationQueueManager->massUpdateTranslationStatus(
            $translationIds,
            TranslationStatus::STATUS_ERROR_ON_PROCESSING,
            'Error while preparing data for translation: ' . $exception->getMessage()
        );
    }

    /**
     * Process the result of the translation
     *
     * @param array $translatedData
     * @param array $storeLanguageMapping
     * @param string $entityType
     * @param int $translationId
     * @return void
     */
    protected function processResult(
        array $translatedData,
        array $storeLanguageMapping,
        string $entityType,
        int $translationId
    ): void {
        if ($translatedData['success'] === false || empty($translatedData['data'])) {
            $this->translationQueueManager->updateTranslationStatus(
                $translationId,
                TranslationStatus::STATUS_ERROR_ON_TRANSLATION,
                $translatedData['error'] ?? 'No data received from translation service'
            );

            return;
        }

        try {
            $translatedData = $translatedData['data'];

            while (isset($translatedData['data'])) {
                $translatedData = $translatedData['data'];
            }

            $this->saveTranslatedData(
                $translatedData,
                $storeLanguageMapping,
                $entityType
            );
        } catch (Exception $exception) {
            $this->translationQueueManager->updateTranslationStatus(
                $translationId,
                TranslationStatus::STATUS_ERROR_ON_SAVING,
                'Error while saving translated data: ' . $exception->getMessage()
            );
            $this->logger->error(
                'Error while saving translated data for translation ID ' . $translationId . ': ' .
                $exception->getMessage() . "\n" . $exception->getTraceAsString()
            );

            return;
        }

        $this->translationQueueManager->updateTranslationStatus(
            $translationId,
            TranslationStatus::STATUS_TRANSLATED,
            'Translation completed'
        );
    }

    /**
     * Process the result of the translation
     * @param array $translatedData
     * @param array $storeLanguageMapping
     * @param string $entityType
     * @return void
     */
    protected function saveTranslatedData(
        array $translatedData,
        array $storeLanguageMapping,
        string $entityType
    ): void {
        foreach ($translatedData as $translatedItems) {
            foreach ($translatedItems as $languageCode => $translatedItem) {
                $storeIds = $storeLanguageMapping[$languageCode];
                $entityTranslator = $this->entityTranslatorFactory->create($entityType);
                unset($translatedItem['translation_id']);

                foreach ($storeIds as $storeId) {
                    $processedTranslatedItem = $translatedItem;

                    if ($this->configHelper->isTermReplacementEnabled()) {
                        $this->termReplacementHelper->replaceForEntity(
                            $entityType,
                            $processedTranslatedItem,
                            $storeId
                        );
                    }

                    $entityTranslator->saveTranslatedData($processedTranslatedItem, [$storeId]);
                }
            }
        }
    }
}
