<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\Translator\Model\Queue;

use Exception;
use GoAi\Translator\Api\TranslationQueueManagerInterface;
use GoAi\Translator\Helper\Config;
use GoAi\Translator\Model\Config\Source\TranslationStatus;
use GoAi\Translator\Model\ResourceModel\Translation\Collection as TranslationCollection;
use GoAi\Translator\Model\TranslatorFactory;
use Magento\Framework\MessageQueue\PublisherInterface;
use Magento\Framework\Serialize\SerializerInterface;
use Magento\Store\Model\Store;
use Psr\Log\LoggerInterface;

/**
 * Abstract Translation Publisher
 */
abstract class AbstractTranslationPublisher
{
    /**
     * TranslationPublisher constructor.
     *
     * @param Config $configHelper
     * @param PublisherInterface $publisher
     * @param SerializerInterface $serializer
     * @param TranslatorFactory $translatorFactory
     * @param TranslationQueueManagerInterface $translationQueueManager
     * @param LoggerInterface $logger
     */
    public function __construct(
        protected Config $configHelper,
        protected PublisherInterface $publisher,
        protected SerializerInterface $serializer,
        protected TranslatorFactory $translatorFactory,
        protected TranslationQueueManagerInterface $translationQueueManager,
        protected LoggerInterface $logger
    ) {
    }

    /**
     * Get topic message queue
     * @return string
     */
    abstract protected function getTopicMessageQueue(): string;

    /**
     * Execute translation publisher
     * @param int $translationType
     * @param string|null $entityType
     * @return void
     */
    public function execute(int $translationType, string|null $entityType = null): void
    {
        if (!$this->configHelper->isTranslatorEnabled()) {
            return;
        }

        try {
            $preparedData = $this->getPreparedData($translationType, $entityType);

            if ($preparedData !== []) {
                $this->publisher->publish(
                    $this->getTopicMessageQueue(),
                    $this->serializer->serialize($preparedData)
                );

                $this->translationQueueManager->massUpdateTranslationStatus(
                    $preparedData['translation_ids'] ?? [],
                    TranslationStatus::STATUS_IN_PROGRESS,
                    'Translation in progress'
                );
            }
        } catch (Exception $exception) {
            $this->logger->critical(
                'Translation Publisher Error: ' . $exception->getMessage(),
                ['exception' => $exception]
            );
            $this->translationQueueManager->massUpdateTranslationStatus(
                $preparedData['translation_ids'] ?? [],
                TranslationStatus::STATUS_ERROR_ON_PROCESSING,
                'Translation error: ' . $exception->getMessage()
            );
        }
    }

    /**
     * Prepare data for translation
     * @param int $translationType
     * @param string|null $entityType
     * @return array
     */
    protected function getPreparedData(int $translationType, string|null $entityType = null): array
    {
        $itemsForTranslation = $this->getItemsForTranslation($translationType, $entityType);
        $translationIds = [];

        foreach ($itemsForTranslation->getItems() as $item) {
            $translationIds[] = $item->getId();
        }

        if ($translationIds === []) {
            return [];
        }

        $storeIds = $this->getStoreIds($itemsForTranslation);
        $languages = [];

        foreach ($storeIds as $storeId) {
            $localeCode = $this->configHelper->getLocaleCode($storeId);
            $languages[$storeId] = $localeCode;
        }

        return [
            'translation_ids' => $translationIds,
            'languages' => $languages
        ];
    }

    /**
     * Get items for translation
     *
     * @param int $translationType
     * @param string|null $entityType
     * @return TranslationCollection
     */
    protected function getItemsForTranslation(int $translationType, string|null $entityType): TranslationCollection
    {
        $translator = $this->translatorFactory->create($translationType);
        return $translator->getItemsForTranslation($entityType);
    }

    /**
     * Get store IDs from items for translation
     * @param TranslationCollection $itemsForTranslation
     * @return array
     */
    protected function getStoreIds(TranslationCollection $itemsForTranslation): array
    {
        $storeIds = [];

        foreach ($itemsForTranslation as $item) {
            $storeIds = array_merge($storeIds, $item->getStoreIds());
        }

        return array_unique($storeIds);
    }
}
