<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\Translator\Controller\Adminhtml\Translation;

use GoAi\Translator\Api\TranslationQueueManagerInterface;
use GoAi\Translator\Helper\Config as ConfigHelper;
use GoAi\Translator\Model\Config\Source\TranslationType;
use Magento\Backend\App\Action;
use Magento\Backend\App\Action\Context;
use Magento\Framework\App\Action\HttpPostActionInterface;
use Magento\Framework\Controller\ResultFactory;
use Magento\Store\Model\StoreManagerInterface;

/**
 * Add Entities to Queue
 */
class AddToQueue extends Action implements HttpPostActionInterface
{
    /**
     * Add To Queue constructor.
     *
     * @param Context $context
     * @param ConfigHelper $configHelper
     * @param TranslationQueueManagerInterface $translationQueueManager
     * @param StoreManagerInterface $storeManager
     */
    public function __construct(
        Context $context,
        protected ConfigHelper $configHelper,
        protected TranslationQueueManagerInterface $translationQueueManager,
        protected StoreManagerInterface $storeManager
    ) {
        parent::__construct($context);
    }

    /**
     * @inheritDoc
     */
    public function execute()
    {
        if (!$this->configHelper->isTranslatorEnabled()) {
            $this->messageManager->addErrorMessage(__('Translation is not enabled.'));
            $resultRedirect = $this->resultFactory->create(ResultFactory::TYPE_REDIRECT);
            return $resultRedirect->setRefererUrl();
        }

        $entityType = $this->getRequest()->getParam('entity_type');
        $storeIds = $this->configHelper->getAvailableStoresByType($entityType);
        $entityIds = $this->getEntityIds();

        if (empty($storeIds)) {
            $this->messageManager->addErrorMessage(__('No stores available for the selected entity type.'));
            $resultRedirect = $this->resultFactory->create(ResultFactory::TYPE_REDIRECT);
            return $resultRedirect->setRefererUrl();
        }

        foreach ($storeIds as $storeId) {
            $this->translationQueueManager->massAddToTranslationQueue(
                $entityType,
                (int) $storeId,
                $entityIds,
                TranslationType::CONTENT_UPDATE
            );
        }

        $this->messageManager->addSuccessMessage(
            count($entityIds) > 1 ?
                __('Items were added to the translation queue.') :
                __('Item was added to the translation queue.')
        );

        $resultRedirect = $this->resultFactory->create(ResultFactory::TYPE_REDIRECT);
        return $resultRedirect->setRefererUrl();
    }

    /**
     * @return array|float[]|int[]|string[]
     */
    protected function getEntityIds(): array
    {
        $entityIds = $this->getRequest()->getParam('selected');

        if (is_string($entityIds) && is_numeric($entityIds)) {
            $entityIds = [$entityIds];
        }

        return $entityIds ?: [];
    }
}
