<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\Translator\Block\Adminhtml\Store;

use GoAi\Translator\Api\TranslationQueueManagerInterface;
use GoAi\Translator\Helper\Config as ConfigHelper;
use GoAi\Translator\Model\System\EntityTypes;
use Magento\Backend\Block\Template;
use Magento\Backend\Block\Template\Context;
use Magento\Directory\Helper\Data as DirectoryHelper;
use Magento\Framework\Json\Helper\Data as JsonHelper;
use Magento\Framework\Registry;

/**
 * Translation Form
 */
class TranslationForm extends Template
{
    /**
     * TranslationForm constructor.
     *
     * @param Context $context
     * @param ConfigHelper $configHelper
     * @param EntityTypes $entityTypes
     * @param Registry $registry
     * @param TranslationQueueManagerInterface $translationQueueManager
     * @param array $data
     * @param JsonHelper|null $jsonHelper
     * @param DirectoryHelper|null $directoryHelper
     */
    public function __construct(
        Context $context,
        protected ConfigHelper $configHelper,
        protected EntityTypes $entityTypes,
        protected Registry $registry,
        protected TranslationQueueManagerInterface $translationQueueManager,
        array $data = [],
        ?JsonHelper $jsonHelper = null,
        ?DirectoryHelper $directoryHelper = null
    ) {
        parent::__construct($context, $data, $jsonHelper, $directoryHelper);
    }

    /**
     * @return int
     */
    public function getStoreViewId(): int
    {
        return (int)$this->registry->registry('store_data')->getId();
    }

    /**
     * @return bool
     */
    public function isStoreViewEditPage(): bool
    {
        return (bool)$this->registry->registry('store_type') &&
            $this->getStoreViewId() !== $this->configHelper->getDefaultStoreId();
    }

    /**
     * @return array
     */
    public function getEnabledEntityTypes(): array
    {
        return $this->configHelper->getEnabledEntityTypes($this->getStoreViewId());
    }

    /**
     * @return int
     */
    public function getTotalEntitiesCount(): int
    {
        return $this->translationQueueManager->getTotalEntitiesCount(
            $this->getStoreViewId()
        );
    }

    /**
     * @return int
     */
    public function getEntitiesCountInQueue(): int
    {
        return $this->translationQueueManager->getEntitiesCountInQueue(
            $this->getStoreViewId()
        );
    }

    /**
     * @param string $entityType
     * @return int
     */
    public function getTotalCountByEntityType(string $entityType): int
    {
        return $this->translationQueueManager->getTotalCountByEntityType($entityType);
    }

    /**
     * Get the total number of entities in the translation queue for a specific entity type.
     * @param string $entityType
     * @return int
     */
    public function getEntitiesCountInQueueByEntityType(
        string $entityType
    ): int {
        return $this->translationQueueManager->getEntitiesCountInQueueByEntityType(
            $this->getStoreViewId(),
            $entityType
        );
    }

    /**
     * Get the entity type title.
     *
     * @param string $entityType
     * @return string
     */
    public function getEntityTypeTitle(
        string $entityType
    ): string {
        return $this->entityTypes->getEntityTypeTitle($entityType);
    }
}
