<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\Translator\Observer;

use GoAi\Translator\Api\TranslationQueueManagerInterface;
use GoAi\Translator\Helper\Config as ConfigHelper;
use GoAi\Translator\Model\EntityTranslator\CmsPage as CmsPageTranslator;
use Magento\Cms\Model\Page;
use Magento\Framework\Event\Observer;
use Magento\Framework\Event\ObserverInterface;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Store\Model\Store;
use Magento\Store\Model\StoreManagerInterface;

/**
 * This class is responsible for handling the before save product event.
 */
class BeforeSaveCmsPage implements ObserverInterface
{
    /**
     * BeforeSaveProduct constructor.
     *
     * @param ConfigHelper $configHelper
     * @param CmsPageTranslator $cmsPageTranslator
     * @param StoreManagerInterface $storeManager
     * @param TranslationQueueManagerInterface $translationManager
     */
    public function __construct(
        protected ConfigHelper $configHelper,
        protected CmsPageTranslator $cmsPageTranslator,
        protected StoreManagerInterface $storeManager,
        protected TranslationQueueManagerInterface $translationManager
    ) {
    }

    /**
     * Handle the before save product event.
     *
     * @param Observer $observer
     * @return void
     * @throws NoSuchEntityException
     */
    public function execute(Observer $observer): void
    {
        if (!$this->configHelper->isTranslateOnEntityChangeEnabled()) {
            return;
        }

        /** @var Page $cmsPage */
        $cmsPage = $observer->getEvent()->getObject();
        $pageStoreIds = $cmsPage->getStores();
        $nonDefaultStores = array_diff(
            $pageStoreIds,
            [$this->cmsPageTranslator->getDefaultStoreId(), Store::DEFAULT_STORE_ID]
        );

        $cmsPage = $observer->getEvent()->getObject();

        if (count($nonDefaultStores) === count($pageStoreIds) || $cmsPage->isObjectNew()) {
            return;
        }

        if ($this->cmsPageTranslator->isUpdated($cmsPage) && !$cmsPage->getIsSkipAiTranslation()) {
            $this->translationManager->addToTranslationQueue(
                CmsPageTranslator::ENTITY_TYPE,
                (int)$cmsPage->getId(),
                $this->configHelper->getAvailableStoresByType(CmsPageTranslator::ENTITY_TYPE)
            );
        }
    }
}
