<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\Translator\Observer;

use GoAi\Translator\Api\TranslationQueueManagerInterface;
use GoAi\Translator\Helper\Config as ConfigHelper;
use GoAi\Translator\Model\EntityTranslator\CmsPage as CmsPageTranslator;
use Magento\Cms\Model\Page;
use Magento\Framework\Event\Observer;
use Magento\Framework\Event\ObserverInterface;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Store\Model\Store;
use Magento\Store\Model\StoreManagerInterface;

/**
 * This class is responsible for handling the after save CMS page event.
 * It handles new CMS pages that don't have an ID during the before save event.
 */
class AfterSaveCmsPage implements ObserverInterface
{
    /**
     * AfterSaveCmsPage constructor.
     *
     * @param ConfigHelper $configHelper
     * @param CmsPageTranslator $cmsPageTranslator
     * @param StoreManagerInterface $storeManager
     * @param TranslationQueueManagerInterface $translationManager
     */
    public function __construct(
        protected ConfigHelper $configHelper,
        protected CmsPageTranslator $cmsPageTranslator,
        protected StoreManagerInterface $storeManager,
        protected TranslationQueueManagerInterface $translationManager
    ) {
    }

    /**
     * Handle the after save CMS page event.
     * This is specifically for handling new CMS pages that don't have an ID during the before save event.
     *
     * @param Observer $observer
     * @return void
     * @throws NoSuchEntityException
     */
    public function execute(Observer $observer): void
    {
        if (!$this->configHelper->isTranslateOnEntityChangeEnabled()) {
            return;
        }

        /** @var Page $cmsPage */
        $cmsPage = $observer->getEvent()->getObject();
        $pageStoreIds = $cmsPage->getStores();
        $nonDefaultStores = array_diff(
            $pageStoreIds,
            [$this->cmsPageTranslator->getDefaultStoreId(), Store::DEFAULT_STORE_ID]
        );

        if (count($nonDefaultStores) === count($pageStoreIds)) {
            return;
        }

        if ($cmsPage->isObjectNew() && !$cmsPage->getIsSkipAiTranslation()) {
            $this->translationManager->addToTranslationQueue(
                CmsPageTranslator::ENTITY_TYPE,
                (int)$cmsPage->getId(),
                $this->configHelper->getAvailableStoresByType(CmsPageTranslator::ENTITY_TYPE)
            );
        }
    }
}
