<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\Translator\Model\EntityTranslator;

use GoAi\Translator\Api\EntityTranslatorInterface;
use GoAi\Translator\Helper\Config as ConfigHelper;
use Magento\Cms\Api\Data\PageInterface;
use Magento\Cms\Api\Data\PageInterfaceFactory as PageFactory;
use Magento\Cms\Api\PageRepositoryInterface;
use Magento\Cms\Model\ResourceModel\Page\CollectionFactory as PageCollectionFactory;
use Magento\Framework\Exception\LocalizedException;
use Magento\Store\Model\Store;
use Magento\Store\Model\StoreManagerInterface;

/**
 * CMS Page Entity Translator
 */
class CmsPage implements EntityTranslatorInterface
{
    /**
     * Entity type
     */
    public const ENTITY_TYPE = 'cms_page';

    /**
     * Skipped attributes
     */
    public const SKIPPED_ATTRIBUTES = [
        PageInterface::IDENTIFIER
    ];

    /**
     * CmsPage constructor.
     *
     * @param ConfigHelper $configHelper
     * @param PageCollectionFactory $collectionFactory
     * @param PageFactory $pageFactory
     * @param PageRepositoryInterface $pageRepository
     * @param StoreManagerInterface $storeManager
     * @param array $translatableFields
     */
    public function __construct(
        protected ConfigHelper $configHelper,
        protected PageCollectionFactory $collectionFactory,
        protected PageFactory $pageFactory,
        protected PageRepositoryInterface $pageRepository,
        protected StoreManagerInterface $storeManager,
        protected array $translatableFields = []
    ) {
    }

    /**
     * @inheritDoc
     */
    public function getEntityIdsForTranslation(int|null $storeId = null): array
    {
        if ($storeId === null) {
            $storeId = $this->getDefaultStoreId();
        }

        $collection = $this->collectionFactory->create()
            ->addStoreFilter($storeId);
        $skippedIds = $this->getSkippedEntityIds();

        if ($skippedIds !== []) {
            $collection->addFieldToFilter(PageInterface::PAGE_ID, ['nin' => $skippedIds]);
        }

        return $collection->getAllIds();
    }

    /**
     * Get entity IDs to skip
     *
     * @return array
     */
    protected function getSkippedEntityIds(): array
    {

        return $this->collectionFactory->create()
            ->addFieldToFilter('is_skip_ai_translation', 1)
            ->getAllIds();
    }

    /**
     * Get translatable attributes
     *
     * @return string[]
     */
    protected function getTranslatableAttributes(): array
    {
        $fieldList = $this->translatableFields;
        $fieldList[] = PageInterface::PAGE_ID;

        return $fieldList;
    }

    /**
     * @inheritDoc
     */
    public function getPreparedEntityData(int $entityId): array
    {
        return $this->getPreparedEntityDataByCmsPage(
            $this->pageRepository->getById($entityId)
        );
    }

    /**
     * @inheritDoc
     */
    public function getPreparedEntityDataByStoreId(int $entityId, int $storeId, bool $isSkipAttributes = false): array
    {
        $preparedEntityData = $this->getPreparedEntityData($entityId);

        if ($isSkipAttributes) {
            foreach (self::SKIPPED_ATTRIBUTES as $attributeCode) {
                unset($preparedEntityData[$attributeCode]);
            }
        }

        return $preparedEntityData;
    }

    /**
     * Get prepared entity data by CMS page
     * @param PageInterface $page
     * @return array
     */
    public function getPreparedEntityDataByCmsPage(PageInterface $page): array
    {
        $preparedData = [];
        $attributes = $this->getTranslatableAttributes();

        foreach ($attributes as $attributeCode) {
            if (empty($page->getData($attributeCode))) {
                continue;
            }

            $preparedData[$attributeCode] = $page->getData($attributeCode);
        }

        return $preparedData;
    }

    /**
     * @inheritDoc
     */
    public function saveTranslatedData(array $entityData, array $storeIds): void
    {
        if ($storeIds === []) {
            return;
        }

        if (!isset($entityData[PageInterface::PAGE_ID])) {
            throw new LocalizedException(
                __('Translation data is missing an ID. Please refresh the status and try again.')
            );
        }

        if (!isset($entityData[PageInterface::IDENTIFIER])) {
            throw new LocalizedException(
                __('Translation data is missing an identifier. Please refresh the status and try again.')
            );
        }

        $entityId = $entityData[PageInterface::PAGE_ID];
        $originalPage = $this->pageRepository->getById($entityId);
        $originalIdentifier = $originalPage->getOriginalIdentifier() ?? $originalPage->getIdentifier();
        $originalPage = $this->updateOriginalPage($originalPage, $storeIds, $originalIdentifier);
        $pageIdentifier = $entityData[PageInterface::IDENTIFIER];

        unset($entityData[PageInterface::IDENTIFIER]);
        unset($entityData[PageInterface::PAGE_ID]);

        foreach ($storeIds as $storeId) {
            $pageCollection = $this->collectionFactory->create()
                ->addFieldToFilter('original_identifier', $originalIdentifier)
                ->addStoreFilter($storeId, false);

            if ($pageCollection->getSize() === 0) {
                $pageCollectionByIdentifier = $this->collectionFactory->create()
                    ->addFieldToFilter(PageInterface::IDENTIFIER, $pageIdentifier)
                    ->addStoreFilter($storeId, false);

                if ($pageCollectionByIdentifier->getSize() > 0) {
                    $pageIdentifier = $this->resolveUniqueIdentifier($pageIdentifier, $storeId);
                }
            }

            /** @var PageInterface $page */
            $page = $pageCollection->getSize() === 0 ?
                $this->pageFactory->create()
                    ->addData($originalPage->getData())
                    ->unsetData('store_id')
                    ->setIdentifier($pageIdentifier)
                    ->setId(null):
                $pageCollection->getFirstItem();

            $page->addData($entityData)
                ->setStores([$storeId])
                ->setStoreId($storeId);

            $this->pageRepository->save($page);
        }
    }

    /**
     * Update original page
     * @param PageInterface $page
     * @param array $storeIds
     * @param string $originalIdentifier
     * @return PageInterface
     * @throws LocalizedException
     */
    protected function updateOriginalPage(PageInterface $page, array $storeIds, string $originalIdentifier): PageInterface
    {
        $originalPageStoreIds = $page->getStores();

        if (in_array(Store::DEFAULT_STORE_ID, $originalPageStoreIds)) {
            $storeList = $this->storeManager->getStores();
            $originalPageStoreIds = [];

            foreach ($storeList as $store) {
                if ($store->getId() !== Store::DEFAULT_STORE_ID) {
                    $originalPageStoreIds[] = $store->getId();
                }
            }
        }

        $intersectedStores = array_intersect($storeIds, $originalPageStoreIds);

        if ($intersectedStores !== [] || in_array(Store::DEFAULT_STORE_ID, $originalPageStoreIds)) {
            $updatedStores = array_diff($originalPageStoreIds, $storeIds);

            $page->setStores($updatedStores !== [] ? $updatedStores : [$this->getDefaultStoreId()]);
            $page->setData('original_identifier', $originalIdentifier)
                ->setData('rewrites_update_force', true);
            $this->pageRepository->save($page);
        }

        return $page;
    }

    /**
     * Resolve URL key to avoid duplicates
     */
    protected function resolveUniqueIdentifier(
        string $identifier,
        int $storeId,
        int $index = 0
    ): string {
        $preparedIdentifier = $index === 0 ? $identifier : $identifier . '-' . $index;
        $index++;

        $collection = $this->collectionFactory->create();
        $collection->addFieldToFilter('identifier', $preparedIdentifier);
        $collection->addStoreFilter($storeId, false);

        return $collection->getSize() ?
            $this->resolveUniqueIdentifier($identifier, $storeId, $index) :
            $preparedIdentifier;
    }

    /**
     * Is Entity Updated
     * @param PageInterface $entity
     * @return bool
     */
    public function isUpdated(PageInterface $entity): bool
    {
        $originalData = $entity->getOrigData();
        $originalCmsPage = $this->pageFactory->create();
        $originalCmsPage->setData($originalData);

        $originalValues = $this->getPreparedEntityDataByCmsPage($originalCmsPage);
        $currentValues = $this->getPreparedEntityDataByCmsPage($entity);

        return crc32(json_encode($originalValues)) !== crc32(json_encode($currentValues));
    }

    /**
     * @inheritDoc
     */
    public function getDefaultStoreId(): int
    {
        return $this->configHelper->getDefaultStoreId();
    }
}
