<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\Translator\Setup\Patch\Data;

use Magento\Catalog\Model\Category;
use Magento\Catalog\Model\Product;
use Magento\Eav\Api\AttributeRepositoryInterface;
use Magento\Eav\Model\Config;
use Magento\Eav\Model\ResourceModel\Entity\Attribute\CollectionFactory as AttributeCollectionFactory;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Setup\ModuleDataSetupInterface;
use Magento\Framework\Setup\Patch\DataPatchInterface;

/**
 * Data Patch to remove 'url_path' from translatable attributes
 */
class RemoveUrlPathFromTranslatableAttributes implements DataPatchInterface
{
    /**
     * Entity types to apply the patch to
     */
    private const TARGET_ENTITY_TYPES = [
        Product::ENTITY,
        Category::ENTITY
    ];

    /**
     * SetIsAvailableForTranslationFlag constructor.
     *
     * @param ModuleDataSetupInterface $moduleDataSetup
     * @param AttributeCollectionFactory $attributeCollectionFactory
     * @param AttributeRepositoryInterface $attributeRepository
     * @param Config $eavConfig
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     */
    public function __construct(
        protected ModuleDataSetupInterface $moduleDataSetup,
        protected AttributeCollectionFactory $attributeCollectionFactory,
        protected AttributeRepositoryInterface $attributeRepository,
        protected Config $eavConfig,
        protected SearchCriteriaBuilder $searchCriteriaBuilder
    ) {
    }

    /**
     * @inheritDoc
     */
    public function apply(): void
    {
        $this->moduleDataSetup->getConnection()->startSetup();

        foreach (self::TARGET_ENTITY_TYPES as $entityTypeCode) {
            $collection = $this->attributeCollectionFactory->create();
            $collection->addFieldToFilter('entity_type_id', ['eq' => $this->getEntityTypeId($entityTypeCode)])
                ->addFieldToFilter('attribute_code', 'url_path');

            foreach ($collection as $attribute) {
                $attribute->setData('is_available_for_translation', 0);
                $this->attributeRepository->save($attribute);
            }
        }

        $this->moduleDataSetup->getConnection()->endSetup();
    }

    /**
     * Get entity type ID by code
     * @param string $entityTypeCode
     * @return int
     * @throws LocalizedException
     */
    protected function getEntityTypeId(string $entityTypeCode): int
    {
        return (int)$this->eavConfig->getEntityType($entityTypeCode)->getId();
    }

    /**
     * @inheritDoc
     */
    public static function getDependencies(): array
    {
        return [];
    }

    /**
     * @inheritDoc
     */
    public function getAliases(): array
    {
        return [];
    }
}
