<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\Translator\Observer;

use GoAi\Translator\Api\TranslationQueueManagerInterface;
use GoAi\Translator\Helper\Config as ConfigHelper;
use GoAi\Translator\Model\EntityTranslator\Category as CategoryTranslator;
use Magento\Catalog\Model\Category;
use Magento\Framework\Event\ObserverInterface;
use Magento\Framework\Event\Observer;
use Magento\Store\Model\Store;
use Magento\Store\Model\StoreManagerInterface;

/**
 * This class is responsible for handling the before save category event.
 */
class BeforeSaveCategory implements ObserverInterface
{
    /**
     * BeforeSaveCategory constructor.
     *
     * @param CategoryTranslator $categoryTranslator
     * @param ConfigHelper $configHelper
     * @param StoreManagerInterface $storeManager
     * @param TranslationQueueManagerInterface $translationManager
     */
    public function __construct(
        protected CategoryTranslator $categoryTranslator,
        protected ConfigHelper $configHelper,
        protected StoreManagerInterface $storeManager,
        protected TranslationQueueManagerInterface $translationManager
    ) {
    }

    /**
     * Handle the before save category event.
     * @param Observer $observer
     * @return void
     */
    public function execute(Observer $observer): void
    {
        if (!$this->configHelper->isTranslateOnEntityChangeEnabled()) {
            return;
        }

        /** @var Category $category */
        $category = $observer->getEvent()->getCategory();
        $storeId = (int)$this->storeManager->getStore()->getId();

        if ($storeId !== $this->categoryTranslator->getDefaultStoreId() || $category->isObjectNew()) {
            return;
        }

        if ($this->categoryTranslator->isUpdated($category) && !$category->getIsSkipAiTranslation()) {
            $this->translationManager->addToTranslationQueue(
                CategoryTranslator::ENTITY_TYPE,
                (int)$category->getId(),
                $this->configHelper->getAvailableStoresByType(CategoryTranslator::ENTITY_TYPE)
            );
        }
    }
}
