<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\Translator\Observer;

use GoAi\Translator\Api\TranslationQueueManagerInterface;
use GoAi\Translator\Helper\Config as ConfigHelper;
use GoAi\Translator\Model\EntityTranslator\Product as ProductTranslator;
use Magento\Catalog\Model\Product;
use Magento\Framework\Event\Observer;
use Magento\Framework\Event\ObserverInterface;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Store\Model\StoreManagerInterface;

/**
 * This class is responsible for handling the after save product event.
 * It handles new products that don't have an ID during the before save event.
 */
class AfterSaveProduct implements ObserverInterface
{
    /**
     * AfterSaveProduct constructor.
     *
     * @param ConfigHelper $configHelper
     * @param ProductTranslator $productTranslator
     * @param StoreManagerInterface $storeManager
     * @param TranslationQueueManagerInterface $translationManager
     */
    public function __construct(
        protected ConfigHelper $configHelper,
        protected ProductTranslator $productTranslator,
        protected StoreManagerInterface $storeManager,
        protected TranslationQueueManagerInterface $translationManager
    ) {
    }

    /**
     * Handle the after save product event.
     * This is specifically for handling new products that don't have an ID during the before save event.
     *
     * @param Observer $observer
     * @return void
     * @throws NoSuchEntityException
     */
    public function execute(Observer $observer): void
    {
        if (!$this->configHelper->isTranslateOnEntityChangeEnabled()) {
            return;
        }

        $storeId = (int)$this->storeManager->getStore()->getId();

        if ($storeId !== $this->productTranslator->getDefaultStoreId()) {
            return;
        }

        /** @var Product $product */
        $product = $observer->getEvent()->getProduct();

        if ($product->isObjectNew() && !$product->getIsSkipAiTranslation()) {
            $this->translationManager->addToTranslationQueue(
                ProductTranslator::ENTITY_TYPE,
                (int)$product->getId(),
                $this->configHelper->getAvailableStoresByType(ProductTranslator::ENTITY_TYPE)
            );
        }
    }
}
