<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\Translator\Observer;

use GoAi\Translator\Api\TranslationQueueManagerInterface;
use GoAi\Translator\Helper\Config as ConfigHelper;
use GoAi\Translator\Model\EntityTranslator\Category as CategoryTranslator;
use Magento\Catalog\Model\Category;
use Magento\Framework\Event\ObserverInterface;
use Magento\Framework\Event\Observer;
use Magento\Store\Model\StoreManagerInterface;

/**
 * This class is responsible for handling the after save category event.
 * It handles new categories that don't have an ID during the before save event.
 */
class AfterSaveCategory implements ObserverInterface
{
    /**
     * AfterSaveCategory constructor.
     *
     * @param CategoryTranslator $categoryTranslator
     * @param ConfigHelper $configHelper
     * @param StoreManagerInterface $storeManager
     * @param TranslationQueueManagerInterface $translationManager
     */
    public function __construct(
        protected CategoryTranslator $categoryTranslator,
        protected ConfigHelper $configHelper,
        protected StoreManagerInterface $storeManager,
        protected TranslationQueueManagerInterface $translationManager
    ) {
    }

    /**
     * Handle the after save category event.
     * This is specifically for handling new categories that don't have an ID during the before save event.
     *
     * @param Observer $observer
     * @return void
     */
    public function execute(Observer $observer): void
    {
        if (!$this->configHelper->isTranslateOnEntityChangeEnabled()) {
            return;
        }

        $storeId = (int)$this->storeManager->getStore()->getId();

        if ($storeId !== $this->categoryTranslator->getDefaultStoreId()) {
            return;
        }

        /** @var Category $category */
        $category = $observer->getEvent()->getCategory();

        if ($category->isObjectNew() && !$category->getIsSkipAiTranslation()) {
            $this->translationManager->addToTranslationQueue(
                CategoryTranslator::ENTITY_TYPE,
                (int)$category->getId(),
                $this->configHelper->getAvailableStoresByType(CategoryTranslator::ENTITY_TYPE)
            );
        }
    }
}
