<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\Translator\Observer;

use GoAi\Translator\Api\TranslationQueueManagerInterface;
use GoAi\Translator\Helper\Config as ConfigHelper;
use GoAi\Translator\Model\EntityTranslator\EavAttribute\ProductEavAttribute as AttributeTranslator;
use Magento\Eav\Model\Entity\Attribute;
use Magento\Framework\Event\ObserverInterface;
use Magento\Framework\Event\Observer;

/**
 * This class is responsible for handling the after save attribute event.
 * It handles new attributes that don't have an ID during the before save event.
 */
class AfterSaveAttribute implements ObserverInterface
{
    /**
     * AfterSaveAttribute constructor.
     *
     * @param ConfigHelper $configHelper
     * @param AttributeTranslator $attributeTranslator
     * @param TranslationQueueManagerInterface $translationManager
     */
    public function __construct(
        protected ConfigHelper $configHelper,
        protected AttributeTranslator $attributeTranslator,
        protected TranslationQueueManagerInterface $translationManager
    ) {
    }

    /**
     * Handle the after save attribute event.
     * This is specifically for handling new attributes that don't have an ID during the before save event.
     *
     * @param Observer $observer
     * @return void
     */
    public function execute(Observer $observer): void
    {
        if (!$this->configHelper->isTranslateOnEntityChangeEnabled()) {
            return;
        }

        /** @var Attribute $attribute */
        $attribute = $observer->getEvent()->getAttribute();
        $isLabelTranslatable = (bool)$attribute->getData('is_label_translatable');

        if ($isLabelTranslatable !== true) {
            return;
        }

        if ($attribute->isObjectNew()) {
            $this->translationManager->addToTranslationQueue(
                AttributeTranslator::ENTITY_TYPE,
                (int)$attribute->getId(),
                $this->configHelper->getAvailableStoresByType(AttributeTranslator::ENTITY_TYPE)
            );
        }
    }
}
