<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\Translator\Model\ResourceModel;

use GoAi\Translator\Api\Data\TranslationInterface;
use GoAi\Translator\Api\Data\TranslationStoreInterface;
use Magento\Framework\EntityManager\EntityManager;
use Magento\Framework\EntityManager\MetadataPool;
use Magento\Framework\Model\AbstractModel;
use Magento\Framework\Model\ResourceModel\Db\AbstractDb;
use Magento\Framework\Model\ResourceModel\Db\Context;
use Magento\Store\Api\Data\StoreInterface;
use Magento\Store\Model\StoreManagerInterface;

/**
 * Translation Resource Model
 */
class Translation extends AbstractDb
{
    /**
     * @var StoreInterface|null $store
     */

    protected StoreInterface|null $store = null;

    /**
     * Translation constructor.
     *
     * @param Context $context
     * @param StoreManagerInterface $storeManager
     * @param EntityManager $entityManager
     * @param MetadataPool $metadataPool
     * @param null $connectionName
     */
    public function __construct(
        Context $context,
        protected StoreManagerInterface $storeManager,
        protected EntityManager $entityManager,
        protected MetadataPool $metadataPool,
        $connectionName = null
    ) {
        parent::__construct($context, $connectionName);
    }

    /**
     * @inheritDoc
     */
    protected function _construct()
    {
        $this->_init(
            TranslationInterface::TABLE_NAME,
            TranslationInterface::TRANSLATION_ID
        );
    }

    /**
     * @inheritDoc
     */
    protected function _afterLoad(AbstractModel $object)
    {
        parent::_afterLoad($object);

        $connection = $this->getConnection();
        $select = $connection->select()
            ->from(
                $this->getTable(TranslationStoreInterface::TABLE_NAME),
                TranslationStoreInterface::STORE_ID
            )->where('translation_id = ?', $object->getTranslationId());

        $storeIds = $connection->fetchCol($select);
        $object->setData(TranslationInterface::STORE_IDS, implode(',', $storeIds));

        return $this;
    }

    /**
     * @inheritDoc
     */
    protected function _afterSave(AbstractModel $object)
    {
        parent::_afterSave($object);

        $this->saveStores($object);

        return $this;
    }

    /**
     * Save stores assigned to the translation
     *
     * @param AbstractModel $object
     */
    protected function saveStores(AbstractModel $object): void
    {
        $connection = $this->getConnection();
        $translationId = (int) $object->getTranslationId();

        // First remove all old records
        $where = ['translation_id = ?' => $translationId];
        $connection->delete($this->getTable(TranslationStoreInterface::TABLE_NAME), $where);

        // Now insert new ones
        $storeIds = $object->getData(TranslationInterface::STORE_IDS);

        if (is_string($storeIds)) {
            $storeIds = explode(',', $storeIds);
        }

        if (!empty($storeIds)) {
            $insertData = [];
            foreach ($storeIds as $storeId) {
                $insertData[] = [
                    TranslationStoreInterface::TRANSLATION_ID => $translationId,
                    TranslationStoreInterface::STORE_ID => (int)$storeId,
                ];
            }

            $connection->insertMultiple(
                $this->getTable(TranslationStoreInterface::TABLE_NAME),
                $insertData
            );
        }
    }
}
