<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\Translator\Helper;

use GoAi\Translator\Helper\Config as ConfigHelper;
use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Framework\App\Helper\Context;

/**
 * Term Replacement Helper
 */
class TermReplacement extends AbstractHelper
{
    /**
     * TermReplacement constructor.
     *
     * @param Context $context
     * @param Config $configHelper
     */
    public function __construct(
        Context $context,
        protected ConfigHelper $configHelper,
    ) {
        parent::__construct($context);
    }

    /**
     * Replace terms in entity data
     * @param string $entityType
     * @param array $entityData
     * @param int $storeId
     * @return array
     */
    public function replaceForEntity(
        string $entityType,
        array $entityData,
        int $storeId
    ): array {
        $rules = $this->configHelper->getTermReplacementRules(
            $entityType,
            $storeId
        );

        if (empty($rules)) {
            return $entityData;
        }

        $processedRules = $this->prepareVocabulary($rules);

        return $this->processItem(
            $processedRules['incorrect_terms'],
            $processedRules['correct_terms'],
            $entityData
        );
    }

    /**
     * @param array $incorrectTerms
     * @param array $correctTerms
     * @param array $itemData
     * @return array
     */
    protected function processItem(array $incorrectTerms, array $correctTerms, array $itemData): array
    {
        foreach ($itemData as &$data) {
            if (is_array($data)) {
                $data = $this->processItem($incorrectTerms, $correctTerms, $data);
                continue;
            }

            $data = str_replace($incorrectTerms, $correctTerms, $data);
        }

        return $itemData;
    }

    public function prepareVocabulary(array $originalRules): array
    {
        $processedRules = [
            'incorrect_terms' => [],
            'correct_terms' => []
        ];

        foreach ($originalRules as $incorrectTerm => $correctTerm) {
            $processedRules['incorrect_terms'][] = $incorrectTerm;
            $processedRules['correct_terms'][] = $correctTerm;
            $incorrectTerm = strtolower($incorrectTerm);
            $correctTerm = strtolower($correctTerm);

            $processedRules['incorrect_terms'][] = strtoupper($incorrectTerm);
            $processedRules['correct_terms'][] = strtoupper($correctTerm);
            $processedRules['incorrect_terms'][] = strtolower($incorrectTerm);
            $processedRules['correct_terms'][] = strtolower($correctTerm);
            $processedRules['incorrect_terms'][] = ucfirst($incorrectTerm);
            $processedRules['correct_terms'][] = ucfirst($correctTerm);
            $processedRules['incorrect_terms'][] = lcfirst($incorrectTerm);
            $processedRules['correct_terms'][] = lcfirst($correctTerm);
            $processedRules['incorrect_terms'][] = ucwords($incorrectTerm);
            $processedRules['correct_terms'][] = ucwords($correctTerm);
        }

        return $processedRules;
    }
}
