<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\Translator\Cron;

use GoAi\Translator\Helper\Config as ConfigHelper;
use GoAi\Translator\Helper\TermReplacement as TermReplacementHelper;
use GoAi\Translator\Model\EntityTranslatorFactory;
use GoAi\Translator\Model\System\EntityTypes;
use Magento\Config\Model\ResourceModel\Config\Data\CollectionFactory;
use Magento\Framework\Serialize\SerializerInterface;

/**
 * Term Replacement Cron
 */
class TermReplacement
{
    /**
     * TermReplacement constructor.
     *
     * @param ConfigHelper $configHelper
     * @param CollectionFactory $collectionFactory
     * @param EntityTranslatorFactory $entityTranslatorFactory
     * @param EntityTypes $entityTypes
     * @param SerializerInterface $serializer
     * @param TermReplacementHelper $termReplacementHelper
     */
    public function __construct(
        protected ConfigHelper $configHelper,
        protected CollectionFactory $collectionFactory,
        protected EntityTranslatorFactory $entityTranslatorFactory,
        protected EntityTypes $entityTypes,
        protected SerializerInterface $serializer,
        protected TermReplacementHelper $termReplacementHelper
    ) {
    }

    /**
     * @return void
     */
    public function execute(): void
    {
        if (!$this->configHelper->isTermReplacementEnabled()) {
            return;
        }

        $entityTypes = $this->entityTypes->getEntityTypes();
        $startOfPreviousDay = date('Y-m-d 00:00:00', strtotime('yesterday'));
        $endOfPreviousDay = date('Y-m-d 23:59:59', strtotime('yesterday'));

        foreach ($entityTypes as $entityType => $entityTypeLabel) {
            $configPath = sprintf(ConfigHelper::XML_PATH_TRANSLATOR_TERM_REPLACEMENT_RULES_PATTERN, $entityType);

            $updatedConfigs = $this->collectionFactory->create()
                ->addFieldToFilter('path', $configPath)
                ->addFieldToFilter('value', ['neq' => '[]'])
                ->addFieldToFilter('updated_at', ['gteq' => $startOfPreviousDay])
                ->addFieldToFilter('updated_at', ['lteq' => $endOfPreviousDay]);

            if (!$updatedConfigs->getSize()) {
                continue;
            }

            foreach ($updatedConfigs as $configData) {
                $storeId = (int)$configData->getScopeId();
                $enabledEntityTypes = $this->configHelper->getEnabledEntityTypesForTermReplacement($storeId);

                if (!in_array($entityType, $enabledEntityTypes)) {
                    continue;
                }

                $entityTranslator = $this->entityTranslatorFactory->create($entityType);
                $entityIds = $entityTranslator->getEntityIdsForTranslation();

                foreach ($entityIds as $entityId) {
                    $entityData = $entityTranslator->getPreparedEntityDataByStoreId(
                        (int)$entityId,
                        $storeId,
                        true
                    );
                    $originalHash = crc32($this->serializer->serialize($entityData));
                    $entityData = $this->termReplacementHelper->replaceForEntity(
                        $entityType,
                        $entityData,
                        $storeId
                    );
                    $updatedHash = crc32($this->serializer->serialize($entityData));

                    if ($originalHash !== $updatedHash) {
                        $entityTranslator->saveTranslatedData($entityData, [$storeId]);
                    }
                }
            }
        }
    }
}
