<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\Translator\Cron;

use Exception;
use GoAi\Translator\Api\Data\TranslationInterface;
use GoAi\Translator\Helper\Config as ConfigHelper;
use GoAi\Translator\Model\Config\Source\TranslationStatus;
use GoAi\Translator\Model\ResourceModel\Translation as TranslationResource;
use GoAi\Translator\Model\ResourceModel\Translation\CollectionFactory as TranslationCollectionFactory;
use Psr\Log\LoggerInterface;
use Zend_Db_Expr;

/**
 * Cleanup class
 */
class Cleanup
{
    /**
     * Cleanup constructor.
     *
     * @param ConfigHelper $configHelper
     * @param TranslationCollectionFactory $collectionFactory
     * @param TranslationResource $translationResource
     * @param LoggerInterface $logger
     */
    public function __construct(
        protected ConfigHelper $configHelper,
        protected TranslationCollectionFactory $collectionFactory,
        protected TranslationResource $translationResource,
        protected LoggerInterface $logger
    ) {
    }

    /**
     * Execute cleanup
     *
     * @return bool
     */
    public function execute(): bool
    {
        if (!$this->configHelper->isTranslatorEnabled()) {
            return true;
        }

        try {
            // Reset "in progress" translations that haven't been updated in 30+ minutes
            $this->resetStaleInProgressTranslations();

            // Delete translated and skipped translations
            $entityIds = $this->collectionFactory->create()
                ->addFieldToFilter(
                    TranslationInterface::STATUS,
                    [
                        'in' => [
                            TranslationStatus::STATUS_TRANSLATED,
                            TranslationStatus::STATUS_SKIPPED
                        ]
                    ]
                )->getAllIds();

            $connection = $this->translationResource->getConnection();
            $tableName = $this->translationResource->getMainTable();

            $connection->delete($tableName, [
                'translation_id IN (?)' => $entityIds
            ]);
        } catch (Exception $exception) {
            $this->logger->critical('Cleanup translation error: ' . $exception->getMessage());
            return false;
        }

        return true;
    }

    /**
     * Reset status of "in progress" translations that haven't been updated in 30+ minutes
     *
     * @return void
     */
    private function resetStaleInProgressTranslations(): void
    {
        try {
            $connection = $this->translationResource->getConnection();
            $tableName = $this->translationResource->getMainTable();

            $connection->update(
                $tableName,
                [TranslationInterface::STATUS => TranslationStatus::STATUS_NEW],
                [
                    TranslationInterface::STATUS . ' = ?' => TranslationStatus::STATUS_IN_PROGRESS,
                    'updated_at <= ?' => new Zend_Db_Expr('DATE_SUB(NOW(), INTERVAL 30 MINUTE)')
                ]
            );
        } catch (Exception $exception) {
            $this->logger->critical('Reset stale in-progress translations error: ' . $exception->getMessage());
        }
    }
}
