<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\Translator\Model\EntityTranslator;

use Exception;
use GoAi\Translator\Api\EntityTranslatorInterface;
use GoAi\Translator\Model\ResourceModel\AttributeLabel as AttributeLabelResource;
use GoAi\Translator\Model\ResourceModel\AttributeOptionValue as AttributeOptionValueResource;
use Magento\Eav\Api\AttributeRepositoryInterface;
use Magento\Eav\Api\Data\AttributeInterface;
use Magento\Eav\Model\AttributeFactory;
use Magento\Eav\Model\Entity\Attribute;
use Magento\Framework\Exception\LocalizedException;
use Magento\Store\Model\Store;

/**
 * Class EavAttribute
 */
abstract class EavAttribute implements EntityTranslatorInterface
{
    /**
     * Entity Type
     */
    public const ENTITY_TYPE = 'eav_attribute';

    /**
     * EavAttribute constructor.
     *
     * @param AttributeFactory $attributeFactory
     * @param AttributeRepositoryInterface $attributeRepository
     * @param AttributeLabelResource $attributeLabelResource
     * @param AttributeOptionValueResource $optionValueResource
     */
    public function __construct(
        protected AttributeFactory $attributeFactory,
        protected AttributeRepositoryInterface $attributeRepository,
        protected AttributeLabelResource $attributeLabelResource,
        protected AttributeOptionValueResource $optionValueResource
    ) {
    }

    /**
     * @param int|null $storeId
     * @inheritDoc
     */
    abstract public function getEntityIdsForTranslation(int|null $storeId = null): array;

    /**
     * @inheritDoc
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    abstract public function getPreparedEntityData(int $entityId): array;

    /**
     * @inheritDoc
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    abstract public function getPreparedEntityDataByStoreId(
        int $entityId,
        int $storeId,
        bool $isSkipAttributes = false
    ): array;

    /**
     * Get prepared entity data by attribute
     */
    protected function getPreparedEntityDataByAttribute(
        AttributeInterface $attribute,
        int $storeId
    ): array {
        $attributeLabels = $attribute->getFrontendLabels();

        $item = [
            'label' => [
                'attribute_id' => $attribute->getAttributeId(),
                'value' => isset($attributeLabels[$storeId]) ?
                    $attributeLabels[$storeId]->getLabel() :
                    $attribute->getDefaultFrontendLabel()
            ],
            'options' => []
        ];

        if ($attribute->usesSource() && $attribute->getSourceModel() === null) {
            $attribute->setStoreId($storeId);
            $options = $attribute->getOptions();

            if ($options === null) {
                return $item;
            }

            foreach ($options as $option) {
                if (!$option->getValue()) {
                    continue;
                }

                $item['options'][] = [
                    'option_id' => $option->getValue(),
                    'value' => $option->getLabel(),
                ];
            }
        }

        return $item;
    }

    /**
     * Check if the attribute is updated
     * @param AttributeInterface $entity
     * @return bool
     */
    public function isUpdated(AttributeInterface $entity): bool
    {
        $originalData = $entity->getOrigData();
        $originalAttribute = $this->attributeFactory->createAttribute(Attribute::class);
        $originalAttribute->setData($originalData);
        $originalLabel = $originalAttribute->getFrontendLabel();
        $newLabel = $entity->getFrontendLabel();

        if (is_array($originalLabel)) {
            $originalLabel = $originalLabel[$this->getDefaultStoreId()] ?? '';
        }

        if (is_array($newLabel)) {
            $newLabel = $newLabel[$this->getDefaultStoreId()] ?? '';
        }

        $originalValues = $this->getPreparedEntityDataByAttribute($originalAttribute, $this->getDefaultStoreId());
        $currentValues = $this->getPreparedEntityDataByAttribute($entity, $this->getDefaultStoreId());

        return crc32(json_encode($originalValues)) !== crc32(json_encode($currentValues)) ||
            $originalLabel !== $newLabel ||
            $entity->dataHasChangedFor('optionvisual')||
            $entity->dataHasChangedFor('serialized_options');
    }

    /**
     * @inheritDoc
     */
    public function saveTranslatedData(array $entityData, array $storeIds): void
    {
        foreach ($storeIds as $storeId) {
            if (isset($entityData['label']) && $entityData['options'] !== []) {
                $this->saveAttributeLabels($entityData['label'], $storeId);
            }

            if (isset($entityData['options']) && $entityData['options'] !== []) {
                $this->saveOptionValueLabels($entityData['options'], $storeId);
            }
        }
    }

    /**
     * Save attribute labels
     *
     * @param array $attributeLabel
     * @param int $storeId
     * @return bool
     * @throws LocalizedException
     */
    protected function saveAttributeLabels(array $attributeLabel, int $storeId): bool
    {
        try {
            $connection = $this->attributeLabelResource->getConnection();
            $tableName = $this->attributeLabelResource->getMainTable();

            $connection->delete($tableName, [
                'store_id = ?' => $storeId,
                'attribute_id IN (?)' => $attributeLabel['attribute_id']
            ]);

            $attributeLabel['store_id'] = $storeId;
            $connection->insertMultiple($tableName, $attributeLabel);

            return true;
        } catch (Exception $exception) {
            throw new LocalizedException(
                __('Unable to save attribute labels: %1', $exception->getMessage())
            );
        }
    }

    /**
     * Save attribute option value labels
     *
     * @param array $valueLabels
     * @param int $storeId
     * @return bool
     * @throws LocalizedException
     */
    protected function saveOptionValueLabels(array $valueLabels, int $storeId): bool
    {
        try {
            $connection = $this->optionValueResource->getConnection();
            $tableName = $this->optionValueResource->getMainTable();

            $optionIds = array_column($valueLabels, 'option_id');
            $connection->delete($tableName, [
                'store_id = ?' => $storeId,
                'option_id IN (?)' => $optionIds
            ]);

            foreach ($valueLabels as &$customOption) {
                $customOption['store_id'] = $storeId;
            }

            $connection->insertMultiple($tableName, $valueLabels);

            return true;
        } catch (Exception $exception) {
            throw new LocalizedException(
                __('Unable to save custom option titles: %1', $exception->getMessage())
            );
        }
    }

    /**
     * @inheritDoc
     */
    public function getDefaultStoreId(): int
    {
        return Store::DEFAULT_STORE_ID;
    }
}
