<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\Translator\Model\EntityTranslator;

use GoAi\Translator\Api\EntityTranslatorInterface;
use GoAi\Translator\Helper\Config as ConfigHelper;
use GoAi\Translator\Model\CategoryManager;
use Magento\Catalog\Api\Data\CategoryInterface;
use Magento\Catalog\Model\Category as CategoryModel;
use Magento\Catalog\Model\CategoryFactory as CategoryFactory;
use Magento\Catalog\Model\ResourceModel\Category\Attribute\CollectionFactory as AttributeCollectionFactory;
use Magento\Catalog\Model\ResourceModel\Category\CollectionFactory as CategoryCollectionFactory;
use Magento\Cms\Api\Data\PageInterface;
use Magento\Framework\Exception\LocalizedException;
use Magento\Store\Model\Store;
use Magento\UrlRewrite\Model\ResourceModel\UrlRewriteCollectionFactory;

/**
 * Class Category
 */
class Category implements EntityTranslatorInterface
{
    /**
     * Default category ID
     */
    protected const DEFAULT_CATEGORY_ID = 2;

    /**
     * Entity ID
     */
    public const ENTITY_ID = 'entity_id';

    /**
     * Entity type
     */
    public const ENTITY_TYPE = 'category';

    /**
     * Skipped attributes
     */
    public const SKIPPED_ATTRIBUTES = [
        'url_key',
        'url_path'
    ];

    /** @var array $translatableAttributes */
    protected array $translatableAttributes = [];

    /**
     * Category constructor.
     *
     * @param AttributeCollectionFactory $attributeCollectionFactory
     * @param CategoryCollectionFactory $categoryCollectionFactory
     * @param CategoryFactory $categoryFactory
     * @param CategoryManager $categoryManager
     * @param ConfigHelper $configHelper
     * @param UrlRewriteCollectionFactory $urlRewriteCollectionFactory
     */
    public function __construct(
        protected AttributeCollectionFactory $attributeCollectionFactory,
        protected CategoryCollectionFactory $categoryCollectionFactory,
        protected CategoryFactory $categoryFactory,
        protected CategoryManager $categoryManager,
        protected ConfigHelper $configHelper,
        protected UrlRewriteCollectionFactory $urlRewriteCollectionFactory
    ) {
    }

    /**
     * @param int|null $storeId
     * @inheritDoc
     */
    public function getEntityIdsForTranslation(int|null $storeId = null): array
    {
        $collection = $this->categoryCollectionFactory->create();
        $skippedIds = $this->getSkippedEntityIds();

        if ($skippedIds !== []) {
            $collection->addFieldToFilter('entity_id', ['nin' => $this->getSkippedEntityIds()]);
        }

        return $collection->getAllIds();
    }

    /**
     * Get entity IDs to skip
     *
     * @return array
     * @throws LocalizedException
     */
    protected function getSkippedEntityIds(): array
    {
        $categoryIds = $this->categoryCollectionFactory->create()
            ->addAttributeToFilter('is_skip_ai_translation', ['eq' => 1])
            ->getAllIds();

        return array_merge(
            $categoryIds,
            [
                CategoryModel::ROOT_CATEGORY_ID,
                self::DEFAULT_CATEGORY_ID,
            ]
        );
    }

    /**
     * Get translatable attributes
     *
     * @return string[]
     */
    protected function getTranslatableAttributes(): array
    {
        if (!empty($this->translatableAttributes)) {
            return $this->translatableAttributes;
        }

        $collection = $this->attributeCollectionFactory->create();
        $collection->addFieldToFilter('is_available_for_translation', 1)
            ->addFieldToFilter('is_global', 0);

        $this->translatableAttributes = $collection->getColumnValues('attribute_code');
        $this->translatableAttributes = array_merge([self::ENTITY_ID], $this->translatableAttributes);

        return $this->translatableAttributes;
    }

    /**
     * @inheritDoc
     */
    public function getPreparedEntityData(int $entityId): array
    {
        return $this->getPreparedEntityDataByStoreId($entityId, $this->getDefaultStoreId());
    }

    /**
     * @inheritDoc
     */
    public function getPreparedEntityDataByStoreId(int $entityId, int $storeId, bool $isSkipAttributes = false): array
    {
        $preparedEntityData = $this->getPreparedEntityDataByCategory(
            $this->getCategory((int)$entityId, (int)$storeId)
        );

        if ($isSkipAttributes) {
            foreach (self::SKIPPED_ATTRIBUTES as $attributeCode) {
                unset($preparedEntityData[$attributeCode]);
            }
        }

        return $preparedEntityData;
    }

    /**
     * Get category by ID
     * @param int $categoryId
     * @param int|null $storeId
     * @return CategoryInterface
     */
    protected function getCategory(int $categoryId, ?int $storeId = null): CategoryInterface
    {
        $category = $this->categoryFactory->create();

        if (null !== $storeId) {
            $category->setStoreId($storeId);
        }

        $category->load($categoryId);

        if (!$category->getId()) {
            throw NoSuchEntityException::singleField('id', $categoryId);
        }

        return $category;
    }

    /**
     * Get prepared entity data for translation
     * @param CategoryInterface $category
     * @return array
     */
    protected function getPreparedEntityDataByCategory(CategoryInterface $category): array
    {
        $preparedData = [];
        $attributes = $this->getTranslatableAttributes();

        foreach ($attributes as $attributeCode) {
            if (empty($category->getData($attributeCode))) {
                continue;
            } elseif ($attributeCode === self::ENTITY_ID) {
                $preparedData[$attributeCode] = $category->getId();
                continue;
            }

            $preparedData[$attributeCode] = $category->getData($attributeCode);
        }

        return $preparedData;
    }

    /**
     * @inheritDoc
     */
    public function saveTranslatedData(array $entityData, array $storeIds): void
    {
        if (!isset($entityData['entity_id'])) {
            throw new LocalizedException(
                __('Translation data is missing an ID. Please refresh the status and try again.')
            );
        }

        $entityId = (int)$entityData['entity_id'];

        foreach ($storeIds as $storeId) {
            $category = $this->getCategory((int)$entityId, (int)$storeId);

            if (isset($entityData['url_key'])) {
                $urlKey = $this->resolveUrlKey($category, $entityData['url_key'], $storeId, $entityId);
                $entityData['url_key'] = $urlKey;

                if (isset($entityData['url_path'])) {
                    $entityData['url_path'] = $urlKey;
                }

                $category->setData('save_rewrites_history', $this->configHelper->isCreateCategoryPermanentRedirect());
            }

            foreach ($this->getTranslatableAttributes() as $attributeCode) {
                if (isset($entityData[$attributeCode])) {
                    $category->setData($attributeCode, $entityData[$attributeCode]);
                }
            }

            $this->categoryManager->save($category, $storeId);
        }
    }

    /**
     * Is Entity Updated
     * @param CategoryInterface $entity
     * @return bool
     */
    public function isUpdated(CategoryInterface $entity): bool
    {
        $originalData = $entity->getOrigData();
        $originalCategory = $this->categoryFactory->create();
        $originalCategory->setData($originalData);

        $originalValues = $this->getPreparedEntityDataByCategory($originalCategory);
        $currentValues = $this->getPreparedEntityDataByCategory($entity);

        return crc32(json_encode($originalValues)) !== crc32(json_encode($currentValues));
    }

    /**
     * Resolve URL key to avoid duplicates
     */
    protected function resolveUrlKey(
        CategoryInterface $category,
        string $urlKey,
        int $storeId,
        int $entityId,
        int $index = 0
    ): string {
        $urlKey = $category->formatUrlKey($urlKey);
        $preparedUrlKey = $index === 0 ? $urlKey : $urlKey . '-' . $index;
        $index++;

        $collection = $this->urlRewriteCollectionFactory->create();
        $collection->addFieldToFilter('request_path', $preparedUrlKey . '.html');
        $collection->addFieldToFilter('entity_type', 'category');
        $collection->addFieldToFilter('entity_id', ['neq' => $entityId]);
        $collection->addFieldToFilter('store_id', $storeId);

        return $collection->getSize() ?
            $this->resolveUrlKey($category, $urlKey, $storeId, $entityId, $index) :
            $preparedUrlKey;
    }

    /**
     * @inheritDoc
     */
    public function getDefaultStoreId(): int
    {
        return Store::DEFAULT_STORE_ID;
    }
}
