<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\Translator\Model;

use Exception;
use GoAi\Translator\Helper\Config as ConfigHelper;
use Magento\Framework\HTTP\Client\Curl;
use Magento\Framework\Serialize\SerializerInterface;
use Psr\Log\LoggerInterface;

/**
 * Translation Sender
 */
class TranslationSender
{
    /**
     * TranslationSender constructor.
     *
     * @param ConfigHelper $configHelper
     * @param Curl $curl
     * @param SerializerInterface $serializer
     * @param LoggerInterface $logger
     */
    public function __construct(
        protected ConfigHelper $configHelper,
        protected Curl $curl,
        protected SerializerInterface $serializer,
        protected LoggerInterface $logger
    ) {
    }

    /**
     * Send request to translation service
     *
     * @param array $query
     * @param array $languages
     * @return array
     */
    public function send(array $query, array $languages): array
    {
        if ($this->configHelper->isTranslatorEnabled() === false) {
            return [
                'success' => false,
                'error' => 'Translation service is not available or some problem on their side.'
            ];
        }

        try {
            $requestBody = $this->getPreparedBody($query, $languages);

            if ($this->configHelper->isEnabledSenderLog()) {
                $this->logger->critical(
                    'Translation request body: ' . serialize($requestBody)
                );
            }

            $this->curl->post($this->configHelper->getTranslatorEndpoint(), $requestBody);

            if ($this->curl->getStatus() !== 200) {
                $message = 'Translation service is not available or some problem on their side. ' .
                    'Status code: ' . $this->curl->getStatus();

                $this->logger->critical($message);

                if ($this->configHelper->isEnabledSenderLog()) {
                    $this->logger->critical($this->curl->getBody());
                }

                return [
                    'success' => false,
                    'error' => 'Translation service is not available or some problem on their side.' .
                        ' Status code: ' . $this->curl->getStatus(),
                ];
            }

            $response = $this->curl->getBody();

            if ($this->configHelper->isEnabledSenderLog()) {
                $this->logger->critical(
                    'Translation response: ' . $response
                );
            }

            return [
                'success' => true,
                'data' => $this->serializer->unserialize($this->curl->getBody())
            ];
        } catch (Exception $exception) {
            $this->logger->critical($exception . ' ' . $exception->getTraceAsString());

            return [
                'success' => false,
                'error' => $exception->getMessage()
            ];
        }
    }

    /**
     * Prepare body for request
     *
     * @param array $query
     * @param array $languages
     * @return string
     */
    public function getPreparedBody(array $query, array $languages): string
    {
        return $this->serializer->serialize(
            [
               'clientUid' => $this->configHelper->getTranslatorCustomerId(),
               'languages' => $languages,
               'excludedTerms' => $this->configHelper->getExcludedTerms(),
               'query' => $query,
           ]
        );
    }
}
