<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\Translator\Observer;

use GoAi\Translator\Api\TranslationQueueManagerInterface;
use GoAi\Translator\Helper\Config as ConfigHelper;
use GoAi\Translator\Model\EntityTranslator\CmsBlock as CmsBlockTranslator;
use Magento\Framework\Event\Observer;
use Magento\Framework\Event\ObserverInterface;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Store\Model\Store;
use Magento\Store\Model\StoreManagerInterface;

/**
 * This class is responsible for handling the before save product event.
 */
class BeforeSaveCmsBlock implements ObserverInterface
{
    /**
     * BeforeSaveProduct constructor.
     *
     * @param ConfigHelper $configHelper
     * @param CmsBlockTranslator $cmsBlockTranslator
     * @param StoreManagerInterface $storeManager
     * @param TranslationQueueManagerInterface $translationManager
     */
    public function __construct(
        protected ConfigHelper $configHelper,
        protected CmsBlockTranslator $cmsBlockTranslator,
        protected StoreManagerInterface $storeManager,
        protected TranslationQueueManagerInterface $translationManager
    ) {
    }

    /**
     * Handle the before save product event.
     *
     * @param Observer $observer
     * @return void
     * @throws NoSuchEntityException
     */
    public function execute(Observer $observer): void
    {
        if (!$this->configHelper->isTranslateOnEntityChangeEnabled()) {
            return;
        }

        $cmsBlock = $observer->getEvent()->getObject();
        $blockStoreIds = $cmsBlock->getStores();
        $nonDefaultStores = array_diff(
            $blockStoreIds,
            [$this->cmsBlockTranslator->getDefaultStoreId(), Store::DEFAULT_STORE_ID]
        );

        if (count($nonDefaultStores) === count($blockStoreIds)) {
            return;
        }

        if ($this->cmsBlockTranslator->isUpdated($cmsBlock) && !$cmsBlock->getIsSkipAiTranslation()) {
            $this->translationManager->addToTranslationQueue(
                CmsBlockTranslator::ENTITY_TYPE,
                (int)$cmsBlock->getId(),
                $this->configHelper->getAvailableStoresByType(CmsBlockTranslator::ENTITY_TYPE)
            );
        }
    }
}
