<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\Translator\Model;

use Exception;
use Magento\Catalog\Api\CategoryRepositoryInterface;
use Magento\Catalog\Api\Data\CategoryInterface;
use Magento\Catalog\Model\Category;
use Magento\Catalog\Model\CategoryRepository\PopulateWithValues;
use Magento\Catalog\Model\ResourceModel\Category as CategoryResource;
use Magento\Framework\Api\ExtensibleDataObjectConverter;
use Magento\Framework\EntityManager\MetadataPool;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Store\Model\StoreManagerInterface;

/**
 * Class CategoryManager
 */
class CategoryManager
{
    /**
     * List of fields that can used config values in case when value does not defined directly
     *
     * @var array
     */
    protected array $useConfigFields = [
        'available_sort_by',
        'default_sort_by',
        'filter_price_range'
    ];

    /**
     * CategoryManager constructor.
     *
     * @param CategoryRepositoryInterface $categoryRepository
     * @param CategoryResource $categoryResource
     * @param ExtensibleDataObjectConverter $extensibleDataObjectConverter
     * @param MetadataPool $metadataPool
     * @param PopulateWithValues $populateWithValues
     * @param StoreManagerInterface $storeManager
     */
    public function __construct(
        protected CategoryRepositoryInterface $categoryRepository,
        protected CategoryResource $categoryResource,
        protected ExtensibleDataObjectConverter $extensibleDataObjectConverter,
        protected MetadataPool $metadataPool,
        protected PopulateWithValues $populateWithValues,
        protected StoreManagerInterface $storeManager
    ) {
    }

    /**
     * Save category
     *
     * @param CategoryInterface $category
     * @param int $storeId
     * @return CategoryInterface
     * @throws CouldNotSaveException
     * @throws NoSuchEntityException
     */
    public function save(
        CategoryInterface $category,
        int $storeId
    ): CategoryInterface {
        $existingData = $this->extensibleDataObjectConverter->toNestedArray(
            $category,
            [],
            CategoryInterface::class
        );
        $existingData = array_diff_key(
            $existingData,
            array_flip([CategoryInterface::KEY_PATH, CategoryInterface::KEY_LEVEL, CategoryInterface::KEY_PARENT_ID])
        );
        $existingData['store_id'] = $storeId;

        if ($category->getId()) {
            $metadata = $this->metadataPool->getMetadata(CategoryInterface::class);

            $category = $this->categoryRepository->get($category->getId(), $storeId);
            $existingData[$metadata->getLinkField()] = $category->getData(
                $metadata->getLinkField()
            );

            if (isset($existingData['image']) && is_array($existingData['image'])) {
                if (!empty($existingData['image']['delete'])) {
                    $existingData['image'] = null;
                } else {
                    if (isset($existingData['image'][0]['name']) && isset($existingData['image'][0]['tmp_name'])) {
                        $existingData['image'] = $existingData['image'][0]['name'];
                    } else {
                        unset($existingData['image']);
                    }
                }
            }
        } else {
            $parentId = $category->getParentId() ?: $this->storeManager->getStore()->getRootCategoryId();
            $parentCategory = $this->categoryRepository->get($parentId, $storeId);
            $existingData[CategoryInterface::KEY_PATH] = $parentCategory->getPath();
            $existingData[CategoryInterface::KEY_PARENT_ID] = $parentId;
            $existingData[CategoryInterface::KEY_LEVEL] = null;
        }

        $this->populateWithValues->execute($category, $existingData);

        try {
            $this->validateCategory($category);
            $this->categoryResource->save($category);
        } catch (Exception $exception) {
            throw new CouldNotSaveException(
                __(
                    'Could not save category: %1',
                    $exception->getMessage()
                ),
                $exception
            );
        }

        return $this->categoryRepository->get($category->getId(), $storeId);
    }

    /**
     * Validate category process
     *
     * @param  Category $category
     * @return void
     * @throws LocalizedException
     */
    protected function validateCategory(Category $category): void
    {
        $useConfigFields = [];

        foreach ($this->useConfigFields as $field) {
            if (!$category->getData($field)) {
                $useConfigFields[] = $field;
            }
        }

        $category->setData('use_post_data_config', $useConfigFields);
        $validate = $category->validate();

        if ($validate !== true) {
            foreach ($validate as $code => $error) {
                if ($error === true) {
                    $attribute = $this->categoryResource->getAttribute($code)->getFrontend()->getLabel();
                    throw new LocalizedException(
                        __('The "%1" attribute is required. Enter and try again.', $attribute)
                    );
                }

                throw new LocalizedException(__($error));
            }
        }

        $category->unsetData('use_post_data_config');
    }
}
