<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\Translator\Console;

use GoAi\Translator\Api\TranslationQueueManagerInterface;
use GoAi\Translator\Helper\Config as ConfigHelper;
use Magento\Framework\Console\Cli;
use Magento\Store\Model\Store;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Exception\InvalidOptionException;

/**
 * Class Run
 */
class Run extends Command
{
    /**
     * Command options
     */
    protected const STORE_ID = 'store-id';
    protected const STORE_IDS = 'store-ids';
    protected const ENTITY_IDS = 'entity-ids';
    protected const ENTITY_TYPE = 'entity-type';

    /**
     * Run constructor.
     *
     * @param ConfigHelper $configHelper
     * @param TranslationQueueManagerInterface $translationQueueManager
     * @param string|null $name
     *
     */
    public function __construct(
        protected ConfigHelper $configHelper,
        protected TranslationQueueManagerInterface $translationQueueManager,
        string $name = null
    ) {
        parent::__construct($name);
    }

    /**
     * @return void
     */
    protected function configure(): void
    {
        $this->setName('goai:translator:run')
            ->setDescription('Run the GoAi Translator command')
            ->addOption(
                self::STORE_ID,
                's',
                InputOption::VALUE_OPTIONAL,
                'Single store ID. Required for onboarding process'
            )
            ->addOption(
                self::STORE_IDS,
                null,
                InputOption::VALUE_OPTIONAL,
                'Comma-separated list of store IDs'
            )
            ->addOption(
                self::ENTITY_IDS,
                null,
                InputOption::VALUE_OPTIONAL,
                'Comma-separated list of entity IDs'
            )
            ->addOption(
                self::ENTITY_TYPE,
                null,
                InputOption::VALUE_OPTIONAL,
                'Entity type'
            );
    }

    /**
     * @param InputInterface $input
     * @param OutputInterface $output
     * @return int
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $storeId = $input->getOption(self::STORE_ID);
        $storeIds = $input->getOption(self::STORE_IDS);
        $entityIds = $input->getOption(self::ENTITY_IDS);
        $entityType = $input->getOption(self::ENTITY_TYPE);

        if ($storeId === null && $entityIds === null) {
            throw new InvalidOptionException(
                'Option --store-id or --entity-ids is required.'
            );
        }

        $anySet = $entityIds || $entityType || $storeIds;
        $allSet = $entityIds && $entityType && $storeIds;

        if ($anySet && !$allSet) {
            throw new InvalidOptionException(
                'Options --store-ids, --entity-ids, and --entity-type must all be provided if any of them is set.'
            );
        }

        if ($entityIds !== null) {
            $entityIds = $entityIds ? explode(',', $entityIds) : [];
            $storeIds = $storeIds ? explode(',', $storeIds) : [];

            foreach ($entityIds as $entityId) {
                $this->translationQueueManager->addToTranslationQueue(
                    $entityType,
                    (int) $entityId,
                    $storeIds
                );
            }

            return Cli::RETURN_SUCCESS;
        }

        $storeId = (int) $storeId;

        if ($storeId === Store::DEFAULT_STORE_ID || $this->configHelper->getDefaultStoreId() === $storeId) {
            throw new InvalidOptionException(
                'You can\'t start onboarding for default store view.'
            );
        }

        $this->translationQueueManager->startOnboarding($storeId);

        return Cli::RETURN_SUCCESS;
    }
}
