<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\Translator\Model\Translator;

use GoAi\Translator\Api\Data\TranslationInterface;
use GoAi\Translator\Api\TranslatorInterface;
use GoAi\Translator\Helper\Config as ConfigHelper;
use GoAi\Translator\Model\Config\Source\TranslationStatus;
use GoAi\Translator\Model\ResourceModel\Translation\Collection as TranslationCollection;
use GoAi\Translator\Model\ResourceModel\Translation\CollectionFactory as TranslationCollectionFactory;

/**
 * Abstract Translator
 */
abstract class AbstractTranslator implements TranslatorInterface
{
    /**
     * SiteOnboarding constructor.
     *
     * @param ConfigHelper $configHelper
     * @param TranslationCollectionFactory $translationCollectionFactory
     */
    public function __construct(
        protected ConfigHelper $configHelper,
        protected TranslationCollectionFactory $translationCollectionFactory
    ) {
    }

    /**
     * @inheritDoc
     */
    public function getItemsForTranslation(string|null $entityType = null): TranslationCollection
    {
        $translationCollection = $this->translationCollectionFactory->create();

        if (!$this->configHelper->isTranslatorEnabled()) {
            return $translationCollection;
        }

        $translationCollection = $this->addTranslatorTypeFilter($translationCollection);

        if ($entityType) {
            $translationCollection = $this->addEntityTypeFilter($translationCollection, $entityType);
        }

        $translationCollection->addFieldToFilter(
            TranslationInterface::STATUS,
            ['eq' => TranslationStatus::STATUS_NEW]
        )->setPageSize(
            $this->configHelper->getTranslatorBatchSize()
        )->setCurPage(1)->setOrder(TranslationInterface::CREATED_AT, 'ASC');

        return $translationCollection;
    }

    /**
     * Add translator type filter to the translation collection
     * @param TranslationCollection $translationCollection
     * @return TranslationCollection
     */
    abstract protected function addTranslatorTypeFilter(
        TranslationCollection $translationCollection
    ): TranslationCollection;

    /**
     * Add entity type filter to the translation collection
     * @param TranslationCollection $translationCollection
     * @param string $entityType
     * @return TranslationCollection
     */
    protected function addEntityTypeFilter(
        TranslationCollection $translationCollection,
        string $entityType
    ): TranslationCollection {
        return $translationCollection->addFieldToFilter(TranslationInterface::ENTITY_TYPE, ['eq' => $entityType]);
    }
}
