<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\Translator\Model\EntityTranslator;

use GoAi\Translator\Api\EntityTranslatorInterface;
use GoAi\Translator\Helper\Config as ConfigHelper;
use Magento\Cms\Api\Data\BlockInterface;
use Magento\Cms\Api\Data\BlockInterfaceFactory as BlockFactory;
use Magento\Cms\Api\BlockRepositoryInterface;
use Magento\Cms\Model\ResourceModel\Block\CollectionFactory as BlockCollectionFactory;
use Magento\Framework\Exception\LocalizedException;
use Magento\Store\Model\Store;
use Magento\Store\Model\StoreManagerInterface;

/**
 * CMS Block Entity Translator
 */
class CmsBlock implements EntityTranslatorInterface
{
    /**
     * Entity type
     */
    public const ENTITY_TYPE = 'cms_block';

    /**
     * Skipped attributes
     */
    public const SKIPPED_ATTRIBUTES = [
        BlockInterface::IDENTIFIER
    ];

    /**
     * CmsBlock constructor.
     *
     * @param ConfigHelper $configHelper
     * @param BlockCollectionFactory $collectionFactory
     * @param BlockFactory $blockFactory
     * @param BlockRepositoryInterface $blockRepository
     * @param StoreManagerInterface $storeManager
     * @param array $translatableFields
     */
    public function __construct(
        protected ConfigHelper $configHelper,
        protected BlockCollectionFactory $collectionFactory,
        protected BlockFactory $blockFactory,
        protected BlockRepositoryInterface $blockRepository,
        protected StoreManagerInterface $storeManager,
        protected array $translatableFields = []
    ) {
    }

    /**
     * @inheritDoc
     */
    public function getEntityIdsForTranslation(int|null $storeId = null): array
    {
        if ($storeId === null) {
            $storeId = $this->getDefaultStoreId();
        }

        $collection = $this->collectionFactory->create()
            ->addStoreFilter($storeId);
        $skippedIds = $this->getSkippedEntityIds();

        if ($skippedIds !== []) {
            $collection->addFieldToFilter(BlockInterface::BLOCK_ID, ['nin' => $skippedIds]);
        }

        return $collection->getAllIds();
    }

    /**
     * Get entity IDs to skip
     *
     * @return array
     */
    protected function getSkippedEntityIds(): array
    {
        return $this->collectionFactory->create()
            ->addFieldToFilter('is_skip_ai_translation', 1)
            ->getAllIds();
    }

    /**
     * Get translatable attributes
     *
     * @return string[]
     */
    protected function getTranslatableAttributes(): array
    {
        $fieldList = $this->translatableFields;
        $fieldList[] = BlockInterface::BLOCK_ID;

        return $fieldList;
    }

    /**
     * @inheritDoc
     */
    public function getPreparedEntityData(int $entityId): array
    {
        return $this->getPreparedEntityDataByCmsBlock(
            $this->blockRepository->getById($entityId)
        );
    }

    /**
     * @inheritDoc
     */
    public function getPreparedEntityDataByStoreId(int $entityId, int $storeId, bool $isSkipAttributes = false): array
    {
        $preparedEntityData = $this->getPreparedEntityData($entityId);

        if ($isSkipAttributes) {
            foreach (self::SKIPPED_ATTRIBUTES as $attributeCode) {
                unset($preparedEntityData[$attributeCode]);
            }
        }

        return $preparedEntityData;
    }

    /**
     * Get prepared entity data by CMS Block
     * @param BlockInterface $block
     * @return array
     */
    public function getPreparedEntityDataByCmsBlock(BlockInterface $block): array
    {
        $preparedData = [];
        $attributes = $this->getTranslatableAttributes();

        foreach ($attributes as $attributeCode) {
            if (empty($block->getData($attributeCode))) {
                continue;
            }

            $preparedData[$attributeCode] = $block->getData($attributeCode);
        }

        return $preparedData;
    }

    /**
     * @inheritDoc
     */
    public function saveTranslatedData(array $entityData, array $storeIds): void
    {
        if ($storeIds === []) {
            return;
        }

        $entityId = $entityData[BlockInterface::BLOCK_ID];
        $originalBlock = $this->blockRepository->getById($entityId);
        $identifier = $originalBlock->getIdentifier();
        $originalBlock = $this->updateOriginalBlock($originalBlock, $storeIds, $identifier);

        unset($entityData[BlockInterface::IDENTIFIER]);
        unset($entityData[BlockInterface::BLOCK_ID]);

        foreach ($storeIds as $storeId) {
            $blockCollection = $this->collectionFactory->create()
                ->addFieldToFilter(BlockInterface::IDENTIFIER, $identifier)
                ->addStoreFilter($storeId, false);

            $block = $blockCollection->getSize() === 0 ?
                $this->blockFactory->create()
                    ->addData($originalBlock->getData())
                    ->unsetData('store_id')
                    ->setId(null):
                $blockCollection->getFirstItem();

            $block->addData($entityData)
                ->setStores([$storeId])
                ->setStoreId($storeId);

            $this->blockRepository->save($block);
        }
    }

    /**
     * Update original block
     * @param BlockInterface $block
     * @param array $storeIds
     * @param string $originalIdentifier
     * @return BlockInterface
     * @throws LocalizedException
     */
    protected function updateOriginalBlock(BlockInterface $block, array $storeIds, string $originalIdentifier): BlockInterface
    {
        $originalBlockStoreIds = $block->getStores();

        if (in_array(Store::DEFAULT_STORE_ID, $originalBlockStoreIds)) {
            $storeList = $this->storeManager->getStores();
            $originalBlockStoreIds = [];

            foreach ($storeList as $store) {
                if ($store->getId() !== Store::DEFAULT_STORE_ID) {
                    $originalBlockStoreIds[] = $store->getId();
                }
            }
        }

        $intersectedStores = array_intersect($storeIds, $originalBlockStoreIds);

        if ($intersectedStores !== []) {
            $updatedStores = array_diff($originalBlockStoreIds, $storeIds);

            $block->setStores($updatedStores !== [] ? $updatedStores : [$this->getDefaultStoreId()]);
            $block->setData('original_identifier', $originalIdentifier)
                ->setData('rewrites_update_force', true);
            $this->blockRepository->save($block);
        }

        return $block;
    }

    /**
     * Resolve URL key to avoid duplicates
     */
    protected function resolveUniqueIdentifier(
        string $identifier,
        int $storeId,
        int $index = 0
    ): string {
        $preparedIdentifier = $index === 0 ? $identifier : $identifier . '-' . $index;
        $index++;

        $collection = $this->collectionFactory->create();
        $collection->addFieldToFilter('identifier', $preparedIdentifier);
        $collection->addStoreFilter($storeId, false);

        return $collection->getSize() ?
            $this->resolveUniqueIdentifier($identifier, $storeId, $index) :
            $preparedIdentifier;
    }

    /**
     * Is Entity Updated
     * @param BlockInterface $entity
     * @return bool
     */
    public function isUpdated(BlockInterface $entity): bool
    {
        $originalData = $entity->getOrigData();
        $originalCmsBlock = $this->blockFactory->create();
        $originalCmsBlock->setData($originalData);

        $originalValues = $this->getPreparedEntityDataByCmsBlock($originalCmsBlock);
        $currentValues = $this->getPreparedEntityDataByCmsBlock($entity);

        return crc32(json_encode($originalValues)) !== crc32(json_encode($currentValues));
    }

    /**
     * @inheritDoc
     */
    public function getDefaultStoreId(): int
    {
        return $this->configHelper->getDefaultStoreId();
    }
}
