<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\Translator\Helper;

use Exception;
use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Framework\App\Helper\Context;
use Magento\Framework\Serialize\SerializerInterface;
use Magento\Store\Model\ScopeInterface;
use Magento\Store\Model\StoreManagerInterface;

/**
 * Config Helper
 */
class Config extends AbstractHelper
{
    /**
     * Config paths
     */
    public const XML_PATH_TRANSLATOR_ENABLED = 'translator/general/enabled';
    public const XML_PATH_TRANSLATOR_ENABLE_TRANSLATION_ON_ENTITY_CHANGE = 'translator/general/translate_on_entity_change';
    public const XML_PATH_TRANSLATOR_DEFAULT_STORE_ID = 'translator/general/default_store_id';
    public const XML_PATH_TRANSLATOR_ENDPOINT = 'translator/general/endpoint';
    public const XML_PATH_TRANSLATOR_BATCH_SIZE = 'translator/general/batch_size';
    public const XML_PATH_TRANSLATOR_CUSTOMER_ID = 'translator/general/customer_id';
    public const XML_PATH_TRANSLATOR_ENABLED_ENTITY_TYPES = 'translator/general/enabled_entity_types';
    public const XML_PATH_TRANSLATOR_CREATE_PERMANENT_REDIRECT_PRODUCT =
        'translator/permanent_redirect_management/product';
    public const XML_PATH_TRANSLATOR_CREATE_PERMANENT_REDIRECT_CATEGORY =
        'translator/permanent_redirect_management/category';
    public const XML_PATH_TRANSLATOR_EXCLUDED_TERMS = 'translator/vocabulary_management/excluded_terms';
    public const XML_PATH_TRANSLATOR_ENABLED_TERM_REPLACEMENT =
        'translator/vocabulary_management/enabled_term_replacement';
    public const XML_PATH_TRANSLATOR_ENABLED_TERM_REPLACEMENT_BY_TYPE =
        'translator/vocabulary_management/enabled_term_replacement_by_type';
    public const XML_PATH_TRANSLATOR_TERM_REPLACEMENT_RULES_PATTERN =
        'translator/vocabulary_management/term_replacement_rules_%s';

    /**
     * Config constructor.
     *
     * @param Context $context
     * @param SerializerInterface $serializer
     * @param StoreManagerInterface $storeManager
     */
    public function __construct(
        Context $context,
        protected SerializerInterface $serializer,
        protected StoreManagerInterface $storeManager
    ) {
        parent::__construct($context);
    }

    /**
     * Check is translator enabled
     * @param string|int|null $scopeCode
     * @param string $scopeType
     * @return bool
     */
    public function isTranslatorEnabled(
        string|int|null $scopeCode = null,
        string $scopeType = ScopeInterface::SCOPE_STORE
    ): bool {
        return (bool)$this->scopeConfig->getValue(
            self::XML_PATH_TRANSLATOR_ENABLED,
            $scopeType,
            $scopeCode
        );
    }

    /**
     * @param string|int|null $scopeCode
     * @param string $scopeType
     * @return bool
     */
    public function isTranslateOnEntityChangeEnabled(
        string|int|null $scopeCode = null,
        string $scopeType = ScopeInterface::SCOPE_STORE
    ): bool {
        return $this->isTranslatorEnabled($scopeCode, $scopeType) &&
            (bool)$this->scopeConfig->getValue(
                self::XML_PATH_TRANSLATOR_ENABLE_TRANSLATION_ON_ENTITY_CHANGE,
                $scopeType,
                $scopeCode
            );
    }

    /**
     * Get default store ID
     * @param string|int|null $scopeCode
     * @param string $scopeType
     * @return int
     */
    public function getDefaultStoreId(
        string|int|null $scopeCode = null,
        string $scopeType = ScopeInterface::SCOPE_STORE
    ): int {
        return (int)$this->scopeConfig->getValue(
            self::XML_PATH_TRANSLATOR_DEFAULT_STORE_ID,
            $scopeType,
            $scopeCode
        );
    }

    /**
     * Get translator endpoint
     * @param string|int|null $scopeCode
     * @param string $scopeType
     * @return string
     */
    public function getTranslatorEndpoint(
        string|int|null $scopeCode = null,
        string $scopeType = ScopeInterface::SCOPE_STORE
    ): string {
        return (string)$this->scopeConfig->getValue(
            self::XML_PATH_TRANSLATOR_ENDPOINT,
            $scopeType,
            $scopeCode
        );
    }

    /**
     * Get translator batch size
     * @param string|int|null $scopeCode
     * @param string $scopeType
     * @return int
     */
    public function getTranslatorBatchSize(
        string|int|null $scopeCode = null,
        string $scopeType = ScopeInterface::SCOPE_STORE
    ): int {
        return (int)$this->scopeConfig->getValue(
            self::XML_PATH_TRANSLATOR_BATCH_SIZE,
            $scopeType,
            $scopeCode
        );
    }

    /**
     * Get translator customer ID
     *
     * @param string|int|null $scopeCode
     * @param string $scopeType
     * @return string
     */
    public function getTranslatorCustomerId(
        string|int|null $scopeCode = null,
        string $scopeType = ScopeInterface::SCOPE_STORE
    ): string {
        return (string)$this->scopeConfig->getValue(
            self::XML_PATH_TRANSLATOR_CUSTOMER_ID,
            $scopeType,
            $scopeCode
        );
    }

    /**
     * Get enabled entity types
     * @param string|int|null $scopeCode
     * @param string $scopeType
     * @return array
     */
    public function getEnabledEntityTypes(
        string|int|null $scopeCode = null,
        string $scopeType = ScopeInterface::SCOPE_STORE
    ): array {
        $enabledEntityTypes = $this->scopeConfig->getValue(
            self::XML_PATH_TRANSLATOR_ENABLED_ENTITY_TYPES,
            $scopeType,
            $scopeCode
        );

        if ($enabledEntityTypes === null) {
            return [];
        }

        $enabledEntityTypes = explode(',', $enabledEntityTypes);

        foreach ($enabledEntityTypes as $key => $value) {
            if ($key === 0 && $value === '') {
                unset($enabledEntityTypes[$key]);
            }
        }

        return $enabledEntityTypes;
    }

    /**
     * Get enabled entity types
     * @param string|int|null $scopeCode
     * @param string $scopeType
     * @return array
     */
    public function getEnabledEntityTypesForTermReplacement(
        string|int|null $scopeCode = null,
        string $scopeType = ScopeInterface::SCOPE_STORE
    ): array {
        $enabledEntityTypes = $this->scopeConfig->getValue(
            self::XML_PATH_TRANSLATOR_ENABLED_TERM_REPLACEMENT_BY_TYPE,
            $scopeType,
            $scopeCode
        );

        if ($enabledEntityTypes === null) {
            return [];
        }

        $enabledEntityTypes = explode(',', $enabledEntityTypes);

        foreach ($enabledEntityTypes as $key => $value) {
            if ($key === 0 && $value === '') {
                unset($enabledEntityTypes[$key]);
            }
        }

        return $enabledEntityTypes;
    }

    /**
     * Get available stores by entity type
     * @param string $type
     * @return array
     */
    public function getAvailableStoresByType(string $type): array
    {
        if (!$this->isTranslatorEnabled()) {
            return [];
        }

        $availableForStores = [];
        $stores = $this->storeManager->getStores();

        foreach ($stores as $store) {
            $enabledEntityTypes = $this->getEnabledEntityTypes($store->getCode());

            if (in_array($type, $enabledEntityTypes)) {
                $availableForStores[] = (int)$store->getId();
            }
        }

        return $availableForStores;
    }

    /**
     * Get Locale code
     * @param string|int|null $scopeCode
     * @param string $scopeType
     * @return string
     */
    public function getLocaleCode(
        string|int|null $scopeCode = null,
        string $scopeType = ScopeInterface::SCOPE_STORE
    ): string {
        return $this->scopeConfig->getValue(
            'general/locale/code',
            $scopeType,
            $scopeCode
        );
    }

    /**
     * Get excluded terms
     * @param string|int|null $scopeCode
     * @param string $scopeType
     * @return array
     */
    public function getExcludedTerms(
        string|int|null $scopeCode = null,
        string $scopeType = ScopeInterface::SCOPE_STORE
    ): array {
        try {
            $excludedTerms = $this->serializer->unserialize(
                $this->scopeConfig->getValue(
                    self::XML_PATH_TRANSLATOR_EXCLUDED_TERMS,
                    $scopeType,
                    $scopeCode
                )
            );

            if (!is_array($excludedTerms)) {
                return [];
            }

            $preparedExcludedTerms = [];

            foreach ($excludedTerms as $excludedTerm) {
                if (!is_array($excludedTerm)) {
                    continue;
                }

                $preparedExcludedTerms[] = $excludedTerm['excluded_term'];
            }

            return $preparedExcludedTerms;
        } catch (Exception) {
            return [];
        }
    }

    /**
     * Check if permanent redirect for product must be created
     * @param string|int|null $scopeCode
     * @param string $scopeType
     * @return bool
     */
    public function isCreateProductPermanentRedirect(
        string|int|null $scopeCode = null,
        string $scopeType = ScopeInterface::SCOPE_STORE
    ): bool {
        return (bool)$this->scopeConfig->getValue(
            self::XML_PATH_TRANSLATOR_CREATE_PERMANENT_REDIRECT_PRODUCT,
            $scopeType,
            $scopeCode
        );
    }

    /**
     * Check if permanent redirect for category must be created
     * @param string|int|null $scopeCode
     * @param string $scopeType
     * @return bool
     */
    public function isCreateCategoryPermanentRedirect(
        string|int|null $scopeCode = null,
        string $scopeType = ScopeInterface::SCOPE_STORE
    ): bool {
        return (bool)$this->scopeConfig->getValue(
            self::XML_PATH_TRANSLATOR_CREATE_PERMANENT_REDIRECT_CATEGORY,
            $scopeType,
            $scopeCode
        );
    }

    /**
     * Check is term replacement enabled
     * @param string|int|null $scopeCode
     * @param string $scopeType
     * @return bool
     */
    public function isTermReplacementEnabled(
        string|int|null $scopeCode = null,
        string $scopeType = ScopeInterface::SCOPE_STORE
    ): bool {
        return $this->isTranslatorEnabled($scopeCode, $scopeType) &&
            (bool)$this->scopeConfig->getValue(
                self::XML_PATH_TRANSLATOR_ENABLED_TERM_REPLACEMENT,
                $scopeType,
                $scopeCode
            );
    }

    /**
     * Get term replacement rules
     * @param string $entityType
     * @param string|int|null $scopeCode
     * @param string $scopeType
     * @return array
     */
    public function getTermReplacementRules(
        string $entityType,
        string|int|null $scopeCode = null,
        string $scopeType = ScopeInterface::SCOPE_STORE
    ): array {
        $serializedRules = $this->scopeConfig->getValue(
            sprintf(self::XML_PATH_TRANSLATOR_TERM_REPLACEMENT_RULES_PATTERN, $entityType),
            $scopeType,
            $scopeCode
        );

        try {
            $rules = $this->serializer->unserialize($serializedRules);
            $preparedRules = [];

            foreach ($rules as $rule) {
                if (!is_array($rule)) {
                    continue;
                }

                $preparedRules[$rule['incorrect_term']] = $rule['correct_term'];
            }

            return $preparedRules;
        } catch (Exception $exception) {
            return [];
        }
    }
}
