<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\Translator\Cron;

use Exception;
use GoAi\Translator\Api\Data\TranslationInterface;
use GoAi\Translator\Helper\Config as ConfigHelper;
use GoAi\Translator\Model\Config\Source\TranslationStatus;
use GoAi\Translator\Model\ResourceModel\Translation as TranslationResource;
use GoAi\Translator\Model\ResourceModel\Translation\CollectionFactory as TranslationCollectionFactory;
use Psr\Log\LoggerInterface;

/**
 * Cleanup class
 */
class Cleanup
{
    /**
     * Cleanup constructor.
     *
     * @param ConfigHelper $configHelper
     * @param TranslationCollectionFactory $collectionFactory
     * @param TranslationResource $translationResource
     * @param LoggerInterface $logger
     */
    public function __construct(
        protected ConfigHelper $configHelper,
        protected TranslationCollectionFactory $collectionFactory,
        protected TranslationResource $translationResource,
        protected LoggerInterface $logger
    ) {
    }

    /**
     * Execute cleanup
     *
     * @return bool
     */
    public function execute(): bool
    {
        if (!$this->configHelper->isTranslatorEnabled()) {
            return true;
        }

        try {
            $entityIds = $this->collectionFactory->create()
                ->addFieldToFilter(
                    TranslationInterface::STATUS,
                    [
                        'in' => [
                            TranslationStatus::STATUS_TRANSLATED,
                            TranslationStatus::STATUS_SKIPPED
                        ]
                    ]
                )->getAllIds();

            $connection = $this->translationResource->getConnection();
            $tableName = $this->translationResource->getMainTable();

            $connection->delete($tableName, [
                'translation_id IN (?)' => $entityIds
            ]);
        } catch (Exception $exception) {
            $this->logger->critical('Cleanup translation error: ' . $exception->getMessage());
            return false;
        }

        return true;
    }
}
