<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\Translator\Model\EntityTranslator;

use GoAi\Translator\Api\EntityTranslatorInterface;
use GoAi\Translator\Helper\Config as ConfigHelper;
use GoAi\Translator\Model\ResourceModel\OptionTitle as OptionTitleResource;
use GoAi\Translator\Model\ResourceModel\OptionTypeTitle as ValueTitleResource;
use Exception;
use Magento\Catalog\Api\Data\ProductInterface;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Catalog\Model\Product\Option\Repository as OptionRepository;
use Magento\Catalog\Model\ProductFactory;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory as ProductCollectionFactory;
use Magento\Catalog\Model\ResourceModel\Product\Attribute\CollectionFactory as AttributeCollectionFactory;
use Magento\Framework\Exception\LocalizedException;
use Magento\Store\Model\Store;
use Magento\UrlRewrite\Model\ResourceModel\UrlRewriteCollectionFactory;

/**
 * Class Product
 */
class Product implements EntityTranslatorInterface
{
    /**
     * Entity ID
     */
    public const ENTITY_ID = 'entity_id';

    /**
     * Entity type
     */
    public const ENTITY_TYPE = 'product';

    /**
     * Skipped attributes
     */
    public const SKIPPED_ATTRIBUTES = [
        'url_key',
        'url_path'
    ];

    /** @var array $translatableAttributes */
    protected array $translatableAttributes = [];

    /**
     * Product constructor.
     *
     * @param AttributeCollectionFactory $attributeCollectionFactory
     * @param ConfigHelper $configHelper
     * @param OptionTitleResource $optionTitleResource
     * @param OptionRepository $optionRepository
     * @param ProductCollectionFactory $collectionFactory
     * @param ProductFactory $productFactory
     * @param ProductRepositoryInterface $productRepository
     * @param ValueTitleResource $valueTitleResource
     * @param UrlRewriteCollectionFactory $urlRewriteCollectionFactory
     */
    public function __construct(
        protected AttributeCollectionFactory $attributeCollectionFactory,
        protected ConfigHelper $configHelper,
        protected OptionTitleResource $optionTitleResource,
        protected OptionRepository $optionRepository,
        protected ProductCollectionFactory $collectionFactory,
        protected ProductFactory $productFactory,
        protected ProductRepositoryInterface $productRepository,
        protected ValueTitleResource $valueTitleResource,
        protected UrlRewriteCollectionFactory $urlRewriteCollectionFactory
    ) {
    }

    /**
     * @param int|null $storeId
     * @inheritDoc
     */
    public function getEntityIdsForTranslation(int|null $storeId = null): array
    {
        return $this->collectionFactory->create()
            ->getAllIds();
    }

    /**
     * Get attributes available for translation.
     *
     * @return string[]
     */
    protected function getTranslatableAttributes(): array
    {
        if (!empty($this->translatableAttributes)) {
            return $this->translatableAttributes;
        }

        $collection = $this->attributeCollectionFactory->create();
        $collection->addFieldToFilter('is_available_for_translation', 1)
            ->addFieldToFilter('is_global', 0);

        $this->translatableAttributes = $collection->getColumnValues('attribute_code');
        $this->translatableAttributes[] = self::ENTITY_ID;

        return $this->translatableAttributes;
    }

    /**
     * @inheritDoc
     */
    public function getPreparedEntityData(int $entityId): array
    {
        return $this->getPreparedEntityDataByStoreId(
            $entityId,
            $this->getDefaultStoreId()
        );
    }

    /**
     * @inheritDoc
     */
    public function getPreparedEntityDataByStoreId(int $entityId, int $storeId, bool $isSkipAttributes = false): array
    {
        $preparedEntityData = $this->getPreparedEntityDataByProduct(
            $this->productRepository->getById($entityId, false, $storeId)
        );

        if ($isSkipAttributes) {
            foreach (self::SKIPPED_ATTRIBUTES as $attributeCode) {
                unset($preparedEntityData[$attributeCode]);
            }
        }

        return $preparedEntityData;
    }

    /**
     * Get prepared entity data for translation
     * @param ProductInterface $product
     * @return array
     */
    protected function getPreparedEntityDataByProduct(ProductInterface $product): array
    {
        $preparedData = [];
        $attributes = $this->getTranslatableAttributes();

        foreach ($attributes as $attributeCode) {
            if (empty($product->getData($attributeCode))) {
                continue;
            } elseif ($attributeCode === self::ENTITY_ID) {
                $preparedData[$attributeCode] = $product->getId();
                continue;
            }

            $preparedData[$attributeCode] = $product->getData($attributeCode);
        }

        return $this->collectCustomOptions($product, $preparedData);
    }

    /**
     * Check if the product is updated
     * @param ProductInterface $entity
     * @return bool
     */
    public function isUpdated(ProductInterface $entity): bool
    {
        $originalData = $entity->getOrigData();
        $originalProduct = $this->productFactory->create();
        $originalProduct->setData($originalData);

        $originalValues = $this->getPreparedEntityDataByProduct($originalProduct);
        $currentValues = $this->getPreparedEntityDataByProduct($entity);

        return crc32(json_encode($originalValues)) !== crc32(json_encode($currentValues));
    }

    /**
     * Collect customizable options
     *
     * @param ProductInterface $product
     * @param array $preparedData
     * @return array
     */
    protected function collectCustomOptions(ProductInterface $product, array $preparedData): array
    {
        $options = $this->optionRepository->getList($product->getSku());
        $preparedData['custom_options'] = [];
        $preparedData['custom_option_values'] = [];

        foreach ($options as $option) {
            $preparedData['custom_options'][] = ['option_id' => $option->getId(), 'title' => $option->getTitle()];

            foreach ($option->getValues() as $value) {
                $preparedData['custom_option_values'][] = [
                    'option_type_id' => $value->getOptionTypeId(),
                    'title' => $value->getTitle()
                ];
            }
        }

        return $preparedData;
    }

    /**
     * @inheritDoc
     */
    public function saveTranslatedData(array $entityData, array $storeIds): void
    {
        $entityId = (int)$entityData[self::ENTITY_ID];

        foreach ($storeIds as $storeId) {
            $product = $this->productRepository->getById($entityId, false, $storeId);
            $product->setData('save_rewrites_history', $this->configHelper->isCreateProductPermanentRedirect());

            if (isset($entityData['url_key'])) {
                $urlKey = $this->resolveUrlKey($entityData['url_key'], $storeId, $entityId);
                $entityData['url_key'] = $urlKey;

                if (isset($entityData['url_path'])) {
                    $entityData['url_path'] = $urlKey;
                }
            }

            foreach ($this->getTranslatableAttributes() as $attributeCode) {
                if (isset($entityData[$attributeCode])) {
                    $product->setData($attributeCode, $entityData[$attributeCode]);
                }
            }

            $product->setOptionsSaved(true);
            $this->productRepository->save($product);

            if (isset($entityData['custom_options'])) {
                // You would implement updateCustomOptions() if needed
                $this->saveOptionTitles($entityData['custom_options'], (int)$product->getStoreId());
            }

            if (isset($entityData['custom_option_values'])) {
                // You would implement updateCustomOptions() if needed
                $this->saveValueTitles($entityData['custom_option_values'], (int)$product->getStoreId());
            }
        }
    }

    /**
     * Save custom option titles
     *
     * @param array $customOptions
     * @param int $storeId
     * @return bool
     * @throws LocalizedException
     */
    protected function saveOptionTitles(array $customOptions, int $storeId): bool
    {
        try {
            $connection = $this->optionTitleResource->getConnection();
            $tableName = $this->optionTitleResource->getMainTable();

            $optionIds = array_column($customOptions, 'option_id');
            $connection->delete($tableName, [
                'store_id = ?' => $storeId,
                'option_id IN (?)' => $optionIds
            ]);

            foreach ($customOptions as &$customOption) {
                $customOption['store_id'] = $storeId;
            }

            $connection->insertMultiple($tableName, $customOptions);

            return true;
        } catch (Exception $exception) {
            throw new LocalizedException(
                __('Unable to save custom option titles: %1', $exception->getMessage())
            );
        }
    }

    /**
     * Save custom option value titles
     *
     * @param array $customOptions
     * @param int $storeId
     * @return bool
     * @throws LocalizedException
     */
    protected function saveValueTitles(array $customOptions, int $storeId): bool
    {
        try {
            $connection = $this->valueTitleResource->getConnection();
            $tableName = $this->valueTitleResource->getMainTable();

            $optionIds = array_column($customOptions, 'option_type_id');
            $connection->delete($tableName, [
                'store_id = ?' => $storeId,
                'option_type_id IN (?)' => $optionIds
            ]);

            foreach ($customOptions as &$customOption) {
                $customOption['store_id'] = $storeId;
            }

            $connection->insertMultiple($tableName, $customOptions);

            return true;
        } catch (Exception $exception) {
            throw new LocalizedException(
                __('Unable to save custom option value titles: %1', $exception->getMessage())
            );
        }
    }

    /**
     * Resolve URL key to avoid duplicates
     */
    protected function resolveUrlKey(
        string $urlKey,
        int $storeId,
        int $entityId,
        int $index = 0
    ): string {
        $preparedUrlKey = $index === 0 ? $urlKey : $urlKey . '-' . $index;
        $index++;

        $collection = $this->urlRewriteCollectionFactory->create();
        $collection->addFieldToFilter('request_path', $preparedUrlKey . '.html');
        $collection->addFieldToFilter('entity_type', 'product');
        $collection->addFieldToFilter('entity_id', ['neq' => $entityId]);
        $collection->addFieldToFilter('store_id', $storeId);

        return $collection->getSize() ? $this->resolveUrlKey($urlKey, $storeId, $entityId, $index) : $preparedUrlKey;
    }

    /**
     * @inheritDoc
     */
    public function getDefaultStoreId(): int
    {
        return Store::DEFAULT_STORE_ID;
    }
}
