<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\Translator\Api;

use GoAi\Translator\Model\Config\Source\TranslationType;

/**
 * Interface TranslationQueueManagerInterface
 */
interface TranslationQueueManagerInterface
{
    /**
     * Starts the onboarding process for a specific store.
     * @param int $storeId
     * @return void
     */
    public function startOnboarding(int $storeId): void;

    /**
     * Adds an entity to the translation queue.
     *
     * @param string $entityType
     * @param int $entityId
     * @param array $storeIds
     * @param int $translationType
     * @return void
     */
    public function addToTranslationQueue(
        string $entityType,
        int $entityId,
        array $storeIds,
        int $translationType = TranslationType::CONTENT_UPDATE
    ): void;

    /**
     * Update translation status.
     *
     * @param int $translationId
     * @param int $status
     * @param string|null $message
     * @return void
     */
    public function updateTranslationStatus(
        int $translationId,
        int $status,
        string|null $message = null
    ): void;

    /**
     * Mass update translation status.
     *
     * @param array $translationIds
     * @param int $status
     * @param string|null $message
     * @return void
     */
    public function massUpdateTranslationStatus(
        array $translationIds,
        int $status,
        string|null $message = null
    ): void;

    /**
     * Get the total number of entities in the translation queue.
     * @param int $storeId
     * @return int
     */
    public function getTotalEntitiesCount(int $storeId): int;

    /**
     * Get the total number of entities in the translation queue for a specific translation type.
     *
     * @param int $storeId
     * @param int $translationType
     * @return int
     */
    public function getTotalCountByEntityType(string $entityType): int;

    /**
     * Get the total number of entities in the translation queue for a specific translation type.
     *
     * @param int $storeId
     * @param int $translationType
     * @return int
     */
    public function getEntitiesCountInQueue(
        int $storeId,
        int $translationType = TranslationType::SITE_ONBOARDING
    ): int;

    /**
     * Get the total number of entities in the translation queue for a specific entity type.
     *
     * @param int $storeId
     * @param string $entityType
     * @param int $translationType
     * @return int
     */
    public function getEntitiesCountInQueueByEntityType(
        int $storeId,
        string $entityType,
        int $translationType = TranslationType::SITE_ONBOARDING
    ): int;

    /**
     * Get the total number of entities in the translation queue for a specific entity type.
     * @param int $defaultStoreId
     * @return void
     */
    public function replaceDefaultStore(int $defaultStoreId): void;
}
