<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\TranslatorMagefanBlog\Observer;

use GoAi\Translator\Api\TranslationQueueManagerInterface;
use GoAi\Translator\Helper\Config as ConfigHelper;
use GoAi\TranslatorMagefanBlog\Model\EntityTranslator\Post as PostTranslator;
use Magento\Framework\Event\Observer;
use Magento\Framework\Event\ObserverInterface;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Store\Model\Store;
use Magento\Store\Model\StoreManagerInterface;

/**
 * This class is responsible for handling the before post save event in the Magefan Blog module.
 */
class BeforeSavePost implements ObserverInterface
{
    /**
     * BeforeSaveProduct constructor.
     *
     * @param ConfigHelper $configHelper
     * @param PostTranslator $postTranslator
     * @param StoreManagerInterface $storeManager
     * @param TranslationQueueManagerInterface $translationManager
     */
    public function __construct(
        protected ConfigHelper $configHelper,
        protected PostTranslator $postTranslator,
        protected StoreManagerInterface $storeManager,
        protected TranslationQueueManagerInterface $translationManager
    ) {
    }

    /**
     * Handle the before save product event.
     *
     * @param Observer $observer
     * @return void
     * @throws NoSuchEntityException
     */
    public function execute(Observer $observer): void
    {
        if (!$this->configHelper->isTranslateOnEntityChangeEnabled()) {
            return;
        }

        $post = $observer->getEvent()->getBlogPost();
        $postStoreIds = $post->getStoreIds();
        $nonDefaultStores = array_diff(
            $postStoreIds,
            [$this->postTranslator->getDefaultStoreId(), Store::DEFAULT_STORE_ID]
        );

        if (count($nonDefaultStores) === count($postStoreIds) || $post->isObjectNew()) {
            return;
        }

        if ($this->postTranslator->isUpdated($post) && !$post->getIsSkipAiTranslation()) {
            $this->translationManager->addToTranslationQueue(
                PostTranslator::ENTITY_TYPE,
                (int)$post->getId(),
                $this->configHelper->getAvailableStoresByType(PostTranslator::ENTITY_TYPE)
            );
        }
    }
}
