<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\TranslatorMagefanBlog\Observer;

use GoAi\Translator\Api\TranslationQueueManagerInterface;
use GoAi\Translator\Helper\Config as ConfigHelper;
use GoAi\TranslatorMagefanBlog\Model\EntityTranslator\Post as PostTranslator;
use Magento\Framework\Event\Observer;
use Magento\Framework\Event\ObserverInterface;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Store\Model\StoreManagerInterface;

/**
 * This class is responsible for handling the after save post event in the Magefan Blog module.
 * It handles new posts that don't have an ID during the before save event.
 */
class AfterSavePost implements ObserverInterface
{
    /**
     * AfterSavePost constructor.
     *
     * @param ConfigHelper $configHelper
     * @param PostTranslator $postTranslator
     * @param StoreManagerInterface $storeManager
     * @param TranslationQueueManagerInterface $translationManager
     */
    public function __construct(
        protected ConfigHelper $configHelper,
        protected PostTranslator $postTranslator,
        protected StoreManagerInterface $storeManager,
        protected TranslationQueueManagerInterface $translationManager
    ) {
    }

    /**
     * Handle the after save post event.
     * This is specifically for handling new posts that don't have an ID during the before save event.
     *
     * @param Observer $observer
     * @return void
     * @throws NoSuchEntityException
     */
    public function execute(Observer $observer): void
    {
        if (!$this->configHelper->isTranslateOnEntityChangeEnabled()) {
            return;
        }

        $post = $observer->getEvent()->getObject();
        $storeId = (int)$this->storeManager->getStore()->getId();

        if ($storeId !== $this->postTranslator->getDefaultStoreId()) {
            return;
        }

        if ((bool)$post->getData('is_skip_ai_translation')) {
            return;
        }

        // Check if this is a new post (isObjectNew is true)
        if ($post->isObjectNew()) {
            $this->translationManager->addToTranslationQueue(
                PostTranslator::ENTITY_TYPE,
                (int)$post->getId(),
                $this->configHelper->getAvailableStoresByType(PostTranslator::ENTITY_TYPE)
            );
        }
    }
}
