<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\TranslatorMagefanBlog\Model\EntityTranslator;

use GoAi\Translator\Api\EntityTranslatorInterface;
use GoAi\Translator\Helper\Config as ConfigHelper;
use Magefan\Blog\Model\ResourceModel\Post as PostResourceModel;
use Magefan\Blog\Model\ResourceModel\Post\CollectionFactory as PostCollectionFactory;
use Magefan\Blog\Model\Tag as TagModel;
use Magefan\Blog\Model\TagFactory;
use Magefan\Blog\Api\TagRepositoryInterface;
use Magefan\Blog\Model\ResourceModel\Tag\CollectionFactory as TagCollectionFactory;
use Magento\Framework\DB\Adapter\AdapterInterface;
use Magento\Framework\Exception\LocalizedException;
use Magento\Store\Model\Store;
use Magento\Store\Model\StoreManagerInterface;

/**
 * Magefan Blog Tag Entity Translator
 */
class Tag implements EntityTranslatorInterface
{
    /**
     * Entity type
     */
    public const ENTITY_TYPE = 'magefan_blog_tag';

    /**
     * Skipped attributes
     */
    public const SKIPPED_ATTRIBUTES = [
        'identifier'
    ];

    /**
     * @var AdapterInterface|false
     */
    protected AdapterInterface|false $connection = false;

    /**
     * Tag constructor.
     *
     * @param ConfigHelper $configHelper
     * @param TagCollectionFactory $collectionFactory
     * @param TagFactory $tagFactory
     * @param TagRepositoryInterface $tagRepository
     * @param StoreManagerInterface $storeManager
     * @param array $translatableFields
     */
    public function __construct(
        protected ConfigHelper $configHelper,
        protected PostCollectionFactory $postCollectionFactory,
        protected PostResourceModel $postResource,
        protected TagCollectionFactory $collectionFactory,
        protected TagFactory $tagFactory,
        protected TagRepositoryInterface $tagRepository,
        protected StoreManagerInterface $storeManager,
        protected array $translatableFields = []
    ) {
    }

    /**
     * @inheritDoc
     */
    public function getEntityIdsForTranslation(?int $storeId = null): array
    {
        if ($storeId === null) {
            $storeId = $this->getDefaultStoreId();
        }

        $collection = $this->collectionFactory->create()
            ->addStoreFilter($storeId);
        $skippedIds = $this->getSkippedEntityIds();

        if ($skippedIds !== []) {
            $collection->addFieldToFilter('tag_id', ['nin' => $skippedIds]);
        }

        return $collection->load()->getAllIds();
    }

    /**
     * Get entity IDs to skip
     *
     * @return array
     */
    protected function getSkippedEntityIds(): array
    {
        return $this->collectionFactory->create()
            ->addFieldToFilter('is_skip_ai_translation', 1)
            ->getAllIds();
    }

    /**
     * Get translatable attributes
     *
     * @return string[]
     */
    protected function getTranslatableAttributes(): array
    {
        $fieldList = $this->translatableFields;
        $fieldList[] = 'tag_id';

        return $fieldList;
    }

    /**
     * @inheritDoc
     */
    public function getPreparedEntityData(int $entityId): array
    {
        return $this->getPreparedEntityDataByTag(
            $this->tagRepository->getById($entityId)
        );
    }

    /**
     * @inheritDoc
     */
    public function getPreparedEntityDataByStoreId(int $entityId, int $storeId, bool $isSkipAttributes = false): array
    {
        $preparedEntityData = $this->getPreparedEntityData($entityId);

        if ($isSkipAttributes) {
            foreach (self::SKIPPED_ATTRIBUTES as $attributeCode) {
                unset($preparedEntityData[$attributeCode]);
            }
        }

        return $preparedEntityData;
    }

    /**
     * Get prepared entity data by Magefan Blog Tag
     *
     * @param TagModel $tag
     * @return array
     */
    public function getPreparedEntityDataByTag(TagModel $tag): array
    {
        $preparedData = [];
        $attributes = $this->getTranslatableAttributes();

        foreach ($attributes as $attributeCode) {
            if (empty($tag->getData($attributeCode))) {
                continue;
            }

            $preparedData[$attributeCode] = $tag->getData($attributeCode);
        }

        return $preparedData;
    }

    /**
     * @inheritDoc
     */
    public function saveTranslatedData(array $entityData, array $storeIds): void
    {
        if ($storeIds === []) {
            return;
        }

        $entityId = $entityData['tag_id'];
        /** @var TagModel $originalTag */
        $originalTag = $this->tagRepository->getById($entityId);
        $originalIdentifier = $originalTag->getIdentifier();
        $originalTag = $this->updateOriginalTag($originalTag, $storeIds, $originalIdentifier);
        $tagIdentifier = $entityData['identifier'];

        unset($entityData['identifier']);
        unset($entityData['tag_id']);
        $postIds = $this->postCollectionFactory->create()
            ->addTagFilter($originalTag)
            ->load()
            ->getAllIds();

        foreach ($storeIds as $storeId) {
            $tagCollection = $this->collectionFactory->create()
                ->addFieldToFilter('original_identifier', $originalIdentifier)
                ->addStoreFilter($storeId, false);

            if ($tagCollection->getSize() === 0) {
                $tagCollectionByIdentifier = $this->collectionFactory->create()
                    ->addFieldToFilter('identifier', $tagIdentifier)
                    ->addStoreFilter($storeId, false);

                if ($tagCollectionByIdentifier->getSize() > 0) {
                    $tagIdentifier = $this->resolveUniqueIdentifier($tagIdentifier, $storeId);
                }
            }

            $tag = $tagCollection->getSize() === 0 ?
                $this->tagFactory->create()
                    ->addData($originalTag->getData())
                    ->unsetData('store_id')
                    ->setIdentifier($tagIdentifier)
                    ->setId(null):
                $tagCollection->getFirstItem();

            $tag->addData($entityData)
                ->setStoreIds([$storeId]);

            $this->tagRepository->save($tag);
            $this->assignTagForPosts(
                (int)$tag->getId(),
                $postIds
            );
        }
    }

    /**
     * Assign tag for posts
     * @param int $tagId
     * @param array $postIds
     * @return void
     */
    protected function assignTagForPosts(int $tagId, array $postIds): void
    {
        $this->getConnection()->insertMultiple(
            $this->postResource->getTable('magefan_blog_post_tag'),
            array_map(
                static fn ($postId) => [
                    'post_id' => $postId,
                    'tag_id' => $tagId
                ],
                $postIds
            )
        );
    }

    /**
     * @return AdapterInterface|bool
     */
    protected function getConnection(): AdapterInterface|bool
    {
        if ($this->connection) {
            return $this->connection;
        }

        $this->connection = $this->postResource->getConnection();

        return $this->connection;
    }

    /**
     * Update original tag
     * @param TagModel $tag
     * @param array $storeIds
     * @param string $originalIdentifier
     * @return TagModel
     * @throws LocalizedException
     */
    protected function updateOriginalTag(TagModel $tag, array $storeIds, string $originalIdentifier): TagModel
    {
        $originalTagStoreIds = $tag->getStoreIds();

        if (in_array(Store::DEFAULT_STORE_ID, $originalTagStoreIds)) {
            $storeList = $this->storeManager->getStores();
            $originalTagStoreIds = [];

            foreach ($storeList as $store) {
                if ($store->getId() !== Store::DEFAULT_STORE_ID) {
                    $originalTagStoreIds[] = $store->getId();
                }
            }
        }

        $intersectedStores = array_intersect($storeIds, $originalTagStoreIds);

        if ($intersectedStores !== [] || in_array(Store::DEFAULT_STORE_ID, $originalTagStoreIds)) {
            $updatedStores = array_diff($originalTagStoreIds, $storeIds);

            $tag->setStoreIds($updatedStores !== [] ? $updatedStores : [$this->getDefaultStoreId()]);
            $tag->setData('original_identifier', $originalIdentifier)
                ->setData('rewrites_update_force', true);
            $this->tagRepository->save($tag);
        }

        return $tag;
    }

    /**
     * Resolve URL key to avoid duplicates
     */
    protected function resolveUniqueIdentifier(
        string $identifier,
        int $storeId,
        int $index = 0
    ): string {
        $preparedIdentifier = $index === 0 ? $identifier : $identifier . '-' . $index;
        $index++;

        $collection = $this->collectionFactory->create();
        $collection->addFieldToFilter('identifier', $preparedIdentifier);
        $collection->addStoreFilter($storeId, false);

        return $collection->getSize() ?
            $this->resolveUniqueIdentifier($identifier, $storeId, $index) :
            $preparedIdentifier;
    }

    /**
     * Is Entity Updated
     * @param TagModel $entity
     * @return bool
     */
    public function isUpdated(TagModel $entity): bool
    {
        $originalData = $entity->getOrigData();
        $originalTag = $this->tagFactory->create();
        $originalTag->setData($originalData);

        $originalValues = $this->getPreparedEntityDataByTag($originalTag);
        $currentValues = $this->getPreparedEntityDataByTag($entity);

        return crc32(json_encode($originalValues)) !== crc32(json_encode($currentValues));
    }

    /**
     * @inheritDoc
     */
    public function getDefaultStoreId(): int
    {
        return $this->configHelper->getDefaultStoreId();
    }
}
