<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\TranslatorMagefanBlog\Model\EntityTranslator;

use GoAi\Translator\Api\EntityTranslatorInterface;
use GoAi\Translator\Helper\Config as ConfigHelper;
use Magefan\Blog\Model\Post as PostModel;
use Magefan\Blog\Model\PostFactory;
use Magefan\Blog\Api\PostRepositoryInterface;
use Magefan\Blog\Model\ResourceModel\Post\CollectionFactory as PostCollectionFactory;
use Magento\Framework\Exception\LocalizedException;
use Magento\Store\Model\Store;
use Magento\Store\Model\StoreManagerInterface;

/**
 * Magefan Blog Post Entity Translator
 */
class Post implements EntityTranslatorInterface
{
    /**
     * Entity type
     */
    public const ENTITY_TYPE = 'magefan_blog_post';

    /**
     * Skipped attributes
     */
    public const SKIPPED_ATTRIBUTES = [
        'identifier'
    ];

    /**
     * Post constructor.
     *
     * @param ConfigHelper $configHelper
     * @param PostCollectionFactory $collectionFactory
     * @param PostFactory $postFactory
     * @param PostRepositoryInterface $postRepository
     * @param StoreManagerInterface $storeManager
     * @param array $translatableFields
     */
    public function __construct(
        protected ConfigHelper $configHelper,
        protected PostCollectionFactory $collectionFactory,
        protected PostFactory $postFactory,
        protected PostRepositoryInterface $postRepository,
        protected StoreManagerInterface $storeManager,
        protected array $translatableFields = []
    ) {
    }

    /**
     * @inheritDoc
     */
    public function getEntityIdsForTranslation(?int $storeId = null): array
    {
        if ($storeId === null) {
            $storeId = $this->getDefaultStoreId();
        }

        $collection = $this->collectionFactory->create()
            ->addStoreFilter($storeId);
        $skippedIds = $this->getSkippedEntityIds();

        if ($skippedIds !== []) {
            $collection->addFieldToFilter('post_id', ['nin' => $skippedIds]);
        }

        return $collection->load()->getAllIds();
    }

    /**
     * Get entity IDs to skip
     *
     * @return array
     */
    protected function getSkippedEntityIds(): array
    {
        return $this->collectionFactory->create()
            ->addFieldToFilter('is_skip_ai_translation', 1)
            ->getAllIds();
    }

    /**
     * Get translatable attributes
     *
     * @return string[]
     */
    protected function getTranslatableAttributes(): array
    {
        $fieldList = $this->translatableFields;
        $fieldList[] = 'post_id';

        return $fieldList;
    }

    /**
     * @inheritDoc
     */
    public function getPreparedEntityData(int $entityId): array
    {
        return $this->getPreparedEntityDataByPost(
            $this->postRepository->getById($entityId)
        );
    }

    /**
     * @inheritDoc
     */
    public function getPreparedEntityDataByStoreId(int $entityId, int $storeId, bool $isSkipAttributes = false): array
    {
        $preparedEntityData = $this->getPreparedEntityData($entityId);

        if ($isSkipAttributes) {
            foreach (self::SKIPPED_ATTRIBUTES as $attributeCode) {
                unset($preparedEntityData[$attributeCode]);
            }
        }

        return $preparedEntityData;
    }

    /**
     * Get prepared entity data by Magefan Blog Post
     *
     * @param PostModel $post
     * @return array
     */
    public function getPreparedEntityDataByPost(PostModel $post): array
    {
        $preparedData = [];
        $attributes = $this->getTranslatableAttributes();

        foreach ($attributes as $attributeCode) {
            if (empty($post->getData($attributeCode))) {
                continue;
            }

            $preparedData[$attributeCode] = $post->getData($attributeCode);
        }

        return $preparedData;
    }

    /**
     * @inheritDoc
     */
    public function saveTranslatedData(array $entityData, array $storeIds): void
    {
        if ($storeIds === []) {
            return;
        }

        $entityId = $entityData['post_id'];
        /** @var PostModel $originalPost */
        $originalPost = $this->postRepository->getById($entityId);
        $originalIdentifier = $originalPost->getIdentifier();
        $originalPost = $this->updateOriginalPost($originalPost, $storeIds, $originalIdentifier);
        $postIdentifier = $entityData['identifier'];

        unset($entityData['identifier']);
        unset($entityData['post_id']);

        foreach ($storeIds as $storeId) {
            $postCollection = $this->collectionFactory->create()
                ->addFieldToFilter('original_identifier', $originalIdentifier)
                ->addStoreFilter($storeId, false);

            if ($postCollection->getSize() === 0) {
                $postCollectionByIdentifier = $this->collectionFactory->create()
                    ->addFieldToFilter('identifier', $postIdentifier)
                    ->addStoreFilter($storeId, false);

                if ($postCollectionByIdentifier->getSize() > 0) {
                    $postIdentifier = $this->resolveUniqueIdentifier($postIdentifier, $storeId);
                }
            }

            $post = $postCollection->getSize() === 0 ?
                $this->postFactory->create()
                    ->addData($originalPost->getData())
                    ->unsetData('store_id')
                    ->setIdentifier($postIdentifier)
                    ->setId(null):
                $postCollection->getFirstItem();

            $post->addData($entityData)
                ->setStoreIds([$storeId]);

            $this->postRepository->save($post);
        }
    }

    /**
     * Update original post
     * @param PostModel $post
     * @param array $storeIds
     * @param string $originalIdentifier
     * @return PostModel
     * @throws LocalizedException
     */
    protected function updateOriginalPost(PostModel $post, array $storeIds, string $originalIdentifier): PostModel
    {
        $originalPostStoreIds = $post->getStoreIds();

        if (in_array(Store::DEFAULT_STORE_ID, $originalPostStoreIds)) {
            $storeList = $this->storeManager->getStores();
            $originalPostStoreIds = [];

            foreach ($storeList as $store) {
                if ($store->getId() !== Store::DEFAULT_STORE_ID) {
                    $originalPostStoreIds[] = $store->getId();
                }
            }
        }

        $intersectedStores = array_intersect($storeIds, $originalPostStoreIds);

        if ($intersectedStores !== [] || in_array(Store::DEFAULT_STORE_ID, $originalPostStoreIds)) {
            $updatedStores = array_diff($originalPostStoreIds, $storeIds);

            $post->setStoreIds($updatedStores !== [] ? $updatedStores : [$this->getDefaultStoreId()]);
            $post->setData('original_identifier', $originalIdentifier)
                ->setData('rewrites_update_force', true);
            $this->postRepository->save($post);
        }

        return $post;
    }

    /**
     * Resolve URL key to avoid duplicates
     */
    protected function resolveUniqueIdentifier(
        string $identifier,
        int $storeId,
        int $index = 0
    ): string {
        $preparedIdentifier = $index === 0 ? $identifier : $identifier . '-' . $index;
        $index++;

        $collection = $this->collectionFactory->create();
        $collection->addFieldToFilter('identifier', $preparedIdentifier);
        $collection->addStoreFilter($storeId, false);

        return $collection->getSize() ?
            $this->resolveUniqueIdentifier($identifier, $storeId, $index) :
            $preparedIdentifier;
    }

    /**
     * Is Entity Updated
     * @param PostModel $entity
     * @return bool
     */
    public function isUpdated(PostModel $entity): bool
    {
        $originalData = $entity->getOrigData();
        $originalPost = $this->postFactory->create();
        $originalPost->setData($originalData);

        $originalValues = $this->getPreparedEntityDataByPost($originalPost);
        $currentValues = $this->getPreparedEntityDataByPost($entity);

        return crc32(json_encode($originalValues)) !== crc32(json_encode($currentValues));
    }

    /**
     * @inheritDoc
     */
    public function getDefaultStoreId(): int
    {
        return $this->configHelper->getDefaultStoreId();
    }
}
