<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\TranslatorMagefanBlog\Model\EntityTranslator;

use GoAi\Translator\Api\EntityTranslatorInterface;
use GoAi\Translator\Helper\Config as ConfigHelper;
use Magefan\Blog\Model\Category as CategoryModel;
use Magefan\Blog\Model\CategoryFactory;
use Magefan\Blog\Api\CategoryRepositoryInterface;
use Magefan\Blog\Model\ResourceModel\Category\CollectionFactory as CategoryCollectionFactory;
use Magefan\Blog\Model\ResourceModel\Post as PostResourceModel;
use Magefan\Blog\Model\ResourceModel\Post\CollectionFactory as PostCollectionFactory;
use Magento\Framework\DB\Adapter\AdapterInterface;
use Magento\Framework\Exception\LocalizedException;
use Magento\Store\Model\Store;
use Magento\Store\Model\StoreManagerInterface;

/**
 * Magefan Blog Category Entity Translator
 */
class Category implements EntityTranslatorInterface
{
    /**
     * Entity type
     */
    public const ENTITY_TYPE = 'magefan_blog_category';

    /**
     * Skipped attributes
     */
    public const SKIPPED_ATTRIBUTES = [
        'identifier'
    ];

    /**
     * @var AdapterInterface|false
     */
    protected AdapterInterface|false $connection = false;

    /**
     * Category constructor.
     *
     * @param ConfigHelper $configHelper
     * @param CategoryCollectionFactory $collectionFactory
     * @param CategoryFactory $categoryFactory
     * @param CategoryRepositoryInterface $categoryRepository
     * @param PostCollectionFactory $postCollectionFactory
     * @param PostResourceModel $postResource
     * @param StoreManagerInterface $storeManager
     * @param array $translatableFields
     */
    public function __construct(
        protected ConfigHelper $configHelper,
        protected CategoryCollectionFactory $collectionFactory,
        protected CategoryFactory $categoryFactory,
        protected CategoryRepositoryInterface $categoryRepository,
        protected PostCollectionFactory $postCollectionFactory,
        protected PostResourceModel $postResource,
        protected StoreManagerInterface $storeManager,
        protected array $translatableFields = []
    ) {
    }

    /**
     * @inheritDoc
     */
    public function getEntityIdsForTranslation(?int $storeId = null): array
    {
        if ($storeId === null) {
            $storeId = $this->getDefaultStoreId();
        }

        $collection = $this->collectionFactory->create()
            ->addStoreFilter($storeId);
        $skippedIds = $this->getSkippedEntityIds();

        if ($skippedIds !== []) {
            $collection->addFieldToFilter('category_id', ['nin' => $skippedIds]);
        }

        return $collection->load()->getAllIds();
    }

    /**
     * Get entity IDs to skip
     *
     * @return array
     */
    protected function getSkippedEntityIds(): array
    {
        return $this->collectionFactory->create()
            ->addFieldToFilter('is_skip_ai_translation', 1)
            ->getAllIds();
    }

    /**
     * Get translatable attributes
     *
     * @return string[]
     */
    protected function getTranslatableAttributes(): array
    {
        $fieldList = $this->translatableFields;
        $fieldList[] = 'category_id';

        return $fieldList;
    }

    /**
     * @inheritDoc
     */
    public function getPreparedEntityData(int $entityId): array
    {
        return $this->getPreparedEntityDataByCategory(
            $this->categoryRepository->getById($entityId)
        );
    }

    /**
     * @inheritDoc
     */
    public function getPreparedEntityDataByStoreId(int $entityId, int $storeId, bool $isSkipAttributes = false): array
    {
        $preparedEntityData = $this->getPreparedEntityData($entityId);

        if ($isSkipAttributes) {
            foreach (self::SKIPPED_ATTRIBUTES as $attributeCode) {
                unset($preparedEntityData[$attributeCode]);
            }
        }

        return $preparedEntityData;
    }

    /**
     * Get prepared entity data by Magefan Blog Category
     *
     * @param CategoryModel $category
     * @return array
     */
    public function getPreparedEntityDataByCategory(CategoryModel $category): array
    {
        $preparedData = [];
        $attributes = $this->getTranslatableAttributes();

        foreach ($attributes as $attributeCode) {
            if (empty($category->getData($attributeCode))) {
                continue;
            }

            $preparedData[$attributeCode] = $category->getData($attributeCode);
        }

        return $preparedData;
    }

    /**
     * @inheritDoc
     */
    public function saveTranslatedData(array $entityData, array $storeIds): void
    {
        if ($storeIds === []) {
            return;
        }

        $entityId = $entityData['category_id'];
        /** @var CategoryModel $originalCategory */
        $originalCategory = $this->categoryRepository->getById($entityId);
        $originalIdentifier = $originalCategory->getIdentifier();
        $originalCategory = $this->updateOriginalCategory($originalCategory, $storeIds, $originalIdentifier);
        $categoryIdentifier = $entityData['identifier'];

        unset($entityData['identifier']);
        unset($entityData['category_id']);

        $postIds = $this->postCollectionFactory->create()
            ->addCategoryFilter($originalCategory)
            ->load()
            ->getAllIds();

        foreach ($storeIds as $storeId) {
            $categoryCollection = $this->collectionFactory->create()
                ->addFieldToFilter('original_identifier', $originalIdentifier)
                ->addStoreFilter($storeId, false);

            if ($categoryCollection->getSize() === 0) {
                $categoryCollectionByIdentifier = $this->collectionFactory->create()
                    ->addFieldToFilter('identifier', $categoryIdentifier)
                    ->addStoreFilter($storeId, false);

                if ($categoryCollectionByIdentifier->getSize() > 0) {
                    $categoryIdentifier = $this->resolveUniqueIdentifier($categoryIdentifier, $storeId);
                }
            }

            $category = $categoryCollection->getSize() === 0 ?
                $this->categoryFactory->create()
                    ->addData($originalCategory->getData())
                    ->unsetData('store_id')
                    ->setIdentifier($categoryIdentifier)
                    ->setId(null):
                $categoryCollection->getFirstItem();

            $category->addData($entityData)
                ->setStoreIds([$storeId]);

            $this->categoryRepository->save($category);
            $this->assignCategoryForPosts(
                (int)$category->getId(),
                $postIds
            );
        }
    }

    /**
     * Assign category for posts
     * @param int $categoryId
     * @param array $postIds
     * @return void
     */
    protected function assignCategoryForPosts(int $categoryId, array $postIds): void
    {
        $this->getConnection()->insertMultiple(
            $this->postResource->getTable('magefan_blog_post_category'),
            array_map(
                static fn ($postId) => [
                    'post_id' => $postId,
                    'category_id' => $categoryId
                ],
                $postIds
            )
        );
    }

    /**
     * @return AdapterInterface|bool
     */
    protected function getConnection(): AdapterInterface|bool
    {
        if ($this->connection) {
            return $this->connection;
        }

        $this->connection = $this->postResource->getConnection();

        return $this->connection;
    }

    /**
     * Update original category
     * @param CategoryModel $category
     * @param array $storeIds
     * @param string $originalIdentifier
     * @return CategoryModel
     * @throws LocalizedException
     */
    protected function updateOriginalCategory(CategoryModel $category, array $storeIds, string $originalIdentifier): CategoryModel
    {
        $originalCategoryStoreIds = $category->getStoreIds();

        if (in_array(Store::DEFAULT_STORE_ID, $originalCategoryStoreIds)) {
            $storeList = $this->storeManager->getStores();
            $originalCategoryStoreIds = [];

            foreach ($storeList as $store) {
                if ($store->getId() !== Store::DEFAULT_STORE_ID) {
                    $originalCategoryStoreIds[] = $store->getId();
                }
            }
        }

        $intersectedStores = array_intersect($storeIds, $originalCategoryStoreIds);

        if ($intersectedStores !== [] || in_array(Store::DEFAULT_STORE_ID, $originalCategoryStoreIds)) {
            $updatedStores = array_diff($originalCategoryStoreIds, $storeIds);

            $category->setStoreIds($updatedStores !== [] ? $updatedStores : [$this->getDefaultStoreId()]);
            $category->setData('original_identifier', $originalIdentifier)
                ->setData('rewrites_update_force', true);
            $this->categoryRepository->save($category);
        }

        return $category;
    }

    /**
     * Resolve URL key to avoid duplicates
     */
    protected function resolveUniqueIdentifier(
        string $identifier,
        int $storeId,
        int $index = 0
    ): string {
        $preparedIdentifier = $index === 0 ? $identifier : $identifier . '-' . $index;
        $index++;

        $collection = $this->collectionFactory->create();
        $collection->addFieldToFilter('identifier', $preparedIdentifier);
        $collection->addStoreFilter($storeId, false);

        return $collection->getSize() ?
            $this->resolveUniqueIdentifier($identifier, $storeId, $index) :
            $preparedIdentifier;
    }

    /**
     * Is Entity Updated
     * @param CategoryModel $entity
     * @return bool
     */
    public function isUpdated(CategoryModel $entity): bool
    {
        $originalData = $entity->getOrigData();
        $originalCategory = $this->categoryFactory->create();
        $originalCategory->setData($originalData);

        $originalValues = $this->getPreparedEntityDataByCategory($originalCategory);
        $currentValues = $this->getPreparedEntityDataByCategory($entity);

        return crc32(json_encode($originalValues)) !== crc32(json_encode($currentValues));
    }

    /**
     * @inheritDoc
     */
    public function getDefaultStoreId(): int
    {
        return $this->configHelper->getDefaultStoreId();
    }
}
