<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\TranslatorAmastyFaq\Block\Adminhtml\Category\Edit\Button;

use Amasty\Faq\Api\CategoryRepositoryInterface;
use Amasty\Faq\Model\Category as CategoryModel;
use GoAi\Translator\Helper\Config as ConfigHelper;
use GoAi\TranslatorAmastyFaq\Model\EntityTranslator\Category;
use Magento\Backend\Model\UrlInterface;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\View\Element\UiComponent\Control\ButtonProviderInterface;
use Magento\Store\Model\Store;

/**
 * Category Translate Button Block
 */
class Translate implements ButtonProviderInterface
{
    public function __construct(
        protected ConfigHelper $configHelper,
        protected CategoryRepositoryInterface $categoryRepository,
        protected RequestInterface $request,
        protected UrlInterface $urlBuilder
    ) {
    }

    public function getButtonData(): array
    {
        if (!$this->configHelper->isTranslatorEnabled()) {
            return [];
        }

        $categoryId = $this->request->getParam('id');

        if ($categoryId === null) {
            return [];
        }

        try {
            $category = $this->categoryRepository->getById($categoryId);

            if ($this->isDisabled($category)) {
                return [];
            }
        } catch (NoSuchEntityException) {
            return [];
        }

        return [
            'label' => __('Translate with AI'),
            'on_click' => 'confirmSetLocation(\'' . __(
                    'Are you sure you want to translate this product?'
                ) . '\', \'' . $this->urlBuilder->getUrl('goai_translator/translation/addToQueue') .
                '\', {data: {selected: [' . $categoryId . '], entity_type: "' . $this->getEntityType() . '"}})',
            'class' => 'action-secondary',
            'sort_order' => 15
        ];
    }

    /**
     * @param int $categoryId
     * @return bool
     */
    protected function isEnabled(int $categoryId): bool
    {
        try {
            return !$this->isDisabled($this->categoryRepository->getById($categoryId));
        } catch (NoSuchEntityException) {
            return false;
        }
    }

    /**
     * @param CategoryModel $category
     * @return bool
     */
    public function isDisabled(CategoryModel $category): bool
    {
        $categoryStoreIds = $category->getStores();

        if (!is_array($categoryStoreIds)) {
            $categoryStoreIds = [(int)$categoryStoreIds];
        }

        return !in_array($this->configHelper->getDefaultStoreId(), $categoryStoreIds) &&
            !in_array(Store::DEFAULT_STORE_ID, $categoryStoreIds);
    }

    /**
     * @return string
     */
    protected function getEntityType(): string
    {
        return Category::ENTITY_TYPE;
    }
}
