<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\TranslatorAmastyFaq\Observer;

use GoAi\Translator\Api\TranslationQueueManagerInterface;
use GoAi\Translator\Helper\Config as ConfigHelper;
use GoAi\TranslatorAmastyFaq\Model\EntityTranslator\Question as QuestionTranslator;
use Magento\Framework\Event\Observer;
use Magento\Framework\Event\ObserverInterface;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Store\Model\Store;
use Magento\Store\Model\StoreManagerInterface;

/**
 * This class is responsible for handling the after question save event in the Amasty Faq module.
 * It handles new questions that don't have an ID during the before save event.
 */
class AfterSaveQuestion implements ObserverInterface
{
    /**
     * AfterSaveQuestion constructor.
     *
     * @param ConfigHelper $configHelper
     * @param QuestionTranslator $questionTranslator
     * @param StoreManagerInterface $storeManager
     * @param TranslationQueueManagerInterface $translationManager
     */
    public function __construct(
        protected ConfigHelper $configHelper,
        protected QuestionTranslator $questionTranslator,
        protected StoreManagerInterface $storeManager,
        protected TranslationQueueManagerInterface $translationManager
    ) {
    }

    /**
     * Handle the after save question event.
     * This is specifically for handling new questions that don't have an ID during the before save event.
     *
     * @param Observer $observer
     * @return void
     * @throws NoSuchEntityException
     */
    public function execute(Observer $observer): void
    {
        if (!$this->configHelper->isTranslateOnEntityChangeEnabled()) {
            return;
        }

        $storeId = (int)$this->storeManager->getStore()->getId();

        if ($storeId !== $this->questionTranslator->getDefaultStoreId()) {
            return;
        }

        $question = $observer->getEvent()->getFaqQuestion();
        $questionStoreIds = $question->getStores();

        if (!is_array($questionStoreIds)) {
            $questionStoreIds = [(int)$questionStoreIds];
        }

        $nonDefaultStores = array_diff(
            $questionStoreIds,
            [$this->questionTranslator->getDefaultStoreId(), Store::DEFAULT_STORE_ID]
        );

        if (count($nonDefaultStores) === count($questionStoreIds)) {
            return;
        }

        // Check if this is a new question (isObjectNew is true)
        if ($question->isObjectNew() && !$question->getIsSkipAiTranslation()) {
            $this->translationManager->addToTranslationQueue(
                QuestionTranslator::ENTITY_TYPE,
                (int)$question->getId(),
                $this->configHelper->getAvailableStoresByType(QuestionTranslator::ENTITY_TYPE)
            );
        }
    }
}
