<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\TranslatorAmastyFaq\Observer;

use GoAi\Translator\Api\TranslationQueueManagerInterface;
use GoAi\Translator\Helper\Config as ConfigHelper;
use GoAi\TranslatorAmastyFaq\Model\EntityTranslator\Category as CategoryTranslator;
use Magento\Framework\Event\Observer;
use Magento\Framework\Event\ObserverInterface;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Store\Model\Store;
use Magento\Store\Model\StoreManagerInterface;

/**
 * This class is responsible for handling the after category save event in the Amasty Faq module.
 * It handles new categories that don't have an ID during the before save event.
 */
class AfterSaveCategory implements ObserverInterface
{
    /**
     * AfterSaveCategory constructor.
     *
     * @param ConfigHelper $configHelper
     * @param CategoryTranslator $categoryTranslator
     * @param StoreManagerInterface $storeManager
     * @param TranslationQueueManagerInterface $translationManager
     */
    public function __construct(
        protected ConfigHelper $configHelper,
        protected CategoryTranslator $categoryTranslator,
        protected StoreManagerInterface $storeManager,
        protected TranslationQueueManagerInterface $translationManager
    ) {
    }

    /**
     * Handle the after save category event.
     * This is specifically for handling new categories that don't have an ID during the before save event.
     *
     * @param Observer $observer
     * @return void
     * @throws NoSuchEntityException
     */
    public function execute(Observer $observer): void
    {
        if (!$this->configHelper->isTranslateOnEntityChangeEnabled()) {
            return;
        }

        $category = $observer->getEvent()->getFaqCategory();
        $categoryStoreIds = $category->getStores();

        if (!is_array($categoryStoreIds)) {
            $categoryStoreIds = [(int)$categoryStoreIds];
        }

        $nonDefaultStores = array_diff(
            $categoryStoreIds,
            [$this->categoryTranslator->getDefaultStoreId(), Store::DEFAULT_STORE_ID]
        );

        if (count($nonDefaultStores) === count($categoryStoreIds)) {
            return;
        }

        // Check if this is a new category (isObjectNew is true)
        if ($category->isObjectNew() && !$category->getIsSkipAiTranslation()) {
            $this->translationManager->addToTranslationQueue(
                CategoryTranslator::ENTITY_TYPE,
                (int)$category->getId(),
                $this->configHelper->getAvailableStoresByType(CategoryTranslator::ENTITY_TYPE)
            );
        }
    }
}
