<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\TranslatorAmastyFaq\Model\EntityTranslator;

use Amasty\Faq\Model\CategoryFactory;
use Amasty\Faq\Api\CategoryRepositoryInterface;
use Amasty\Faq\Model\ResourceModel\Category\CollectionFactory as CategoryCollectionFactory;
use Amasty\Faq\Model\ResourceModel\Question\CollectionFactory as QuestionCollectionFactory;
use Exception;
use GoAi\Translator\Api\EntityTranslatorInterface;
use GoAi\Translator\Helper\Config as ConfigHelper;
use Amasty\Faq\Model\Category as CategoryModel;
use Magento\Framework\App\Area;
use Magento\Framework\App\State;
use Magento\Framework\Exception\LocalizedException;
use Magento\Store\Model\Store;
use Magento\Store\Model\StoreManagerInterface;

/**
 * Amasty FAQ Category Entity Translator
 */
class Category implements EntityTranslatorInterface
{
    /**
     * Entity type
     */
    public const ENTITY_TYPE = 'amasty_faq_category';

    /**
     * Skipped attributes
     */
    public const SKIPPED_ATTRIBUTES = [
        'url_key'
    ];

    /**
     * Category constructor.
     *
     * @param ConfigHelper $configHelper
     * @param CategoryCollectionFactory $collectionFactory
     * @param CategoryFactory $categoryFactory
     * @param CategoryRepositoryInterface $categoryRepository
     * @param QuestionCollectionFactory $questionCollectionFactory
     * @param State $state
     * @param StoreManagerInterface $storeManager
     * @param array $translatableFields
     * @throws LocalizedException
     */
    public function __construct(
        protected ConfigHelper $configHelper,
        protected CategoryCollectionFactory $collectionFactory,
        protected CategoryFactory $categoryFactory,
        protected CategoryRepositoryInterface $categoryRepository,
        protected QuestionCollectionFactory $questionCollectionFactory,
        protected State $state,
        protected StoreManagerInterface $storeManager,
        protected array $translatableFields = []
    ) {
        try {
            $this->state->getAreaCode();
        } catch (Exception) {
            $this->state->setAreaCode(Area::AREA_ADMINHTML);
        }
    }

    /**
     * @inheritDoc
     */
    public function getEntityIdsForTranslation(?int $storeId = null): array
    {
        if ($storeId === null) {
            $storeId = $this->getDefaultStoreId();
        }

        $collection = $this->collectionFactory->create()
            ->addStoreFilter($storeId);
        $skippedIds = $this->getSkippedEntityIds();

        if ($skippedIds !== []) {
            $collection->addFieldToFilter('category_id', ['nin' => $skippedIds]);
        }

        return $collection->getAllIds();
    }

    /**
     * Get entity IDs to skip
     *
     * @return array
     */
    protected function getSkippedEntityIds(): array
    {
        return $this->collectionFactory->create()
            ->addFieldToFilter('is_skip_ai_translation', 1)
            ->getAllIds();
    }

    /**
     * Get translatable attributes
     *
     * @return string[]
     */
    protected function getTranslatableAttributes(): array
    {
        $fieldList = $this->translatableFields;
        $fieldList[] = 'category_id';

        return $fieldList;
    }

    /**
     * @inheritDoc
     */
    public function getPreparedEntityData(int $entityId): array
    {
        return $this->getPreparedEntityDataByCategory(
            $this->categoryRepository->getById($entityId)
        );
    }

    /**
     * @inheritDoc
     */
    public function getPreparedEntityDataByStoreId(int $entityId, int $storeId, bool $isSkipAttributes = false): array
    {
        $preparedEntityData = $this->getPreparedEntityData($entityId);

        if ($isSkipAttributes) {
            foreach (self::SKIPPED_ATTRIBUTES as $attributeCode) {
                unset($preparedEntityData[$attributeCode]);
            }
        }

        return $preparedEntityData;
    }

    /**
     * Get prepared entity data by Amasty FAQ Category
     *
     * @param CategoryModel $category
     * @return array
     */
    public function getPreparedEntityDataByCategory(CategoryModel $category): array
    {
        $preparedData = [];
        $attributes = $this->getTranslatableAttributes();

        foreach ($attributes as $attributeCode) {
            if (empty($category->getData($attributeCode))) {
                continue;
            }

            $preparedData[$attributeCode] = $category->getData($attributeCode);
        }

        return $preparedData;
    }

    /**
     * @inheritDoc
     */
    public function saveTranslatedData(array $entityData, array $storeIds): void
    {
        if ($storeIds === []) {
            return;
        }

        if (!isset($entityData['category_id'])) {
            throw new LocalizedException(
                __('Translation data is missing an ID. Please refresh the status and try again.')
            );
        }

        $entityId = $entityData['category_id'];
        /** @var CategoryModel $originalCategory */
        $originalCategory = $this->categoryRepository->getById($entityId);
        $originalIdentifier = $originalCategory->getUrlKey();
        $originalCategory = $this->updateOriginalCategory($originalCategory, $storeIds, $originalIdentifier);
        $categoryIdentifier = $entityData['url_key'];

        unset($entityData['url_key']);
        unset($entityData['category_id']);

        $questionIds = $this->questionCollectionFactory->create()
            ->addCategoryFilter([$originalCategory->getId()])
            ->load()
            ->getAllIds();

        foreach ($storeIds as $storeId) {
            $categoryCollection = $this->collectionFactory->create()
                ->addFieldToFilter('original_url_key', $originalIdentifier)
                ->addStoreFilter($storeId, false);

            if ($categoryCollection->getSize() === 0) {
                $categoryCollectionByIdentifier = $this->collectionFactory->create()
                    ->addFieldToFilter('url_key', $categoryIdentifier)
                    ->addStoreFilter($storeId, false);

                if ($categoryCollectionByIdentifier->getSize() > 0) {
                    $categoryIdentifier = $this->resolveUniqueIdentifier($categoryIdentifier, $storeId);
                }
            }

            $categoryNumber = $categoryCollection->getSize();
            $category = $categoryNumber === 0 ?
                $this->categoryFactory->create()
                    ->addData($originalCategory->getData())
                    ->unsetData('store_id')
                    ->setUrlKey($categoryIdentifier)
                    ->setId(null)
                    ->setQuestions(null):
                $categoryCollection->getFirstItem();

            $category->addData($entityData)
                ->setStores([$storeId]);

            $this->categoryRepository->save($category);

            if ($categoryNumber === 0) {
                $this->assignCategoryForQuestions(
                    $category,
                    (int)$category->getId(),
                    $questionIds
                );
            }
        }
    }

    /**
     * Assign category for questions
     *
     * @param CategoryModel $category
     * @param int $categoryId
     * @param array $postIds
     * @return void
     */
    protected function assignCategoryForQuestions(CategoryModel $category, int $categoryId, array $questionIds): void
    {
        $categoryQuestionIds = explode(',', $category->getQuestions($questionIds) ?? '');
        $questionIds = array_filter(array_unique(array_merge($questionIds, $categoryQuestionIds)));
        $category->setQuestions(implode(',', $questionIds));

        $this->categoryRepository->save($category);
    }

    /**
     * Update original category
     * @param CategoryModel $category
     * @param array $storeIds
     * @param string $originalIdentifier
     * @return CategoryModel
     * @throws LocalizedException
     */
    protected function updateOriginalCategory(CategoryModel $category, array $storeIds, string $originalIdentifier): CategoryModel
    {
        $originalCategoryStoreIds = $category->getStores();

        if (!is_array($originalCategoryStoreIds)) {
            $originalCategoryStoreIds = [(int)$originalCategoryStoreIds];
        }

        if (in_array(Store::DEFAULT_STORE_ID, $originalCategoryStoreIds)) {
            $storeList = $this->storeManager->getStores();
            $originalCategoryStoreIds = [];

            foreach ($storeList as $store) {
                if ($store->getId() !== Store::DEFAULT_STORE_ID) {
                    $originalCategoryStoreIds[] = $store->getId();
                }
            }
        }

        $intersectedStores = array_intersect($storeIds, $originalCategoryStoreIds);

        if ($intersectedStores !== [] || in_array(Store::DEFAULT_STORE_ID, $originalCategoryStoreIds)) {
            $updatedStores = array_diff($originalCategoryStoreIds, $storeIds);

            $category->setStores($updatedStores !== [] ? $updatedStores : [$this->getDefaultStoreId()]);
            $category->setData('original_url_key', $originalIdentifier)
                ->setData('rewrites_update_force', true);
            $this->categoryRepository->save($category);
        }

        return $category;
    }

    /**
     * Resolve URL key to avoid duplicates
     */
    protected function resolveUniqueIdentifier(
        string $identifier,
        int $storeId,
        int $index = 0
    ): string {
        $preparedIdentifier = $index === 0 ? $identifier : $identifier . '-' . $index;
        $index++;

        $collection = $this->collectionFactory->create();
        $collection->addFieldToFilter('url_key', $preparedIdentifier);
        $collection->addStoreFilter($storeId, false);

        return $collection->getSize() ?
            $this->resolveUniqueIdentifier($identifier, $storeId, $index) :
            $preparedIdentifier;
    }

    /**
     * Is Entity Updated
     * @param CategoryModel $entity
     * @return bool
     */
    public function isUpdated(CategoryModel $entity): bool
    {
        $originalData = $entity->getOrigData();
        $originalCategory = $this->categoryFactory->create();
        $originalCategory->setData($originalData);

        $originalValues = $this->getPreparedEntityDataByCategory($originalCategory);
        $currentValues = $this->getPreparedEntityDataByCategory($entity);

        return crc32(json_encode($originalValues)) !== crc32(json_encode($currentValues));
    }

    /**
     * @inheritDoc
     */
    public function getDefaultStoreId(): int
    {
        return $this->configHelper->getDefaultStoreId();
    }
}
