<?php
/**
 * @copyright 2025 GoMage
 * All rights reserved
 */
declare(strict_types=1);

namespace GoAi\TranslatorAmastyFaq\Model\EntityTranslator;

use Amasty\Faq\Model\QuestionFactory;
use Amasty\Faq\Api\QuestionRepositoryInterface;
use Amasty\Faq\Model\ResourceModel\Question\CollectionFactory as QuestionCollectionFactory;
use GoAi\Translator\Api\EntityTranslatorInterface;
use GoAi\Translator\Helper\Config as ConfigHelper;
use Amasty\Faq\Model\Question as QuestionModel;
use Magento\Framework\Exception\LocalizedException;
use Magento\Store\Model\Store;
use Magento\Store\Model\StoreManagerInterface;

/**
 * Amasty FAQ Question Entity Translator
 */
class Question implements EntityTranslatorInterface
{
    /**
     * Entity type
     */
    public const ENTITY_TYPE = 'amasty_faq_question';

    /**
     * Skipped attributes
     */
    public const SKIPPED_ATTRIBUTES = [
        'url_key'
    ];

    /**
     * Question constructor.
     *
     * @param ConfigHelper $configHelper
     * @param QuestionCollectionFactory $collectionFactory
     * @param QuestionFactory $questionFactory
     * @param QuestionRepositoryInterface $questionRepository
     * @param StoreManagerInterface $storeManager
     * @param array $translatableFields
     */
    public function __construct(
        protected ConfigHelper $configHelper,
        protected QuestionCollectionFactory $collectionFactory,
        protected QuestionFactory $questionFactory,
        protected QuestionRepositoryInterface $questionRepository,
        protected StoreManagerInterface $storeManager,
        protected array $translatableFields = []
    ) {
    }

    /**
     * @inheritDoc
     */
    public function getEntityIdsForTranslation(?int $storeId = null): array
    {
        if ($storeId === null) {
            $storeId = $this->getDefaultStoreId();
        }

        $collection = $this->collectionFactory->create()
            ->addStoreFilter($storeId);
        $skippedIds = $this->getSkippedEntityIds();

        if ($skippedIds !== []) {
            $collection->addFieldToFilter('question_id', ['nin' => $skippedIds]);
        }

        return $collection->getAllIds();
    }

    /**
     * Get entity IDs to skip
     *
     * @return array
     */
    protected function getSkippedEntityIds(): array
    {
        return $this->collectionFactory->create()
            ->addFieldToFilter('is_skip_ai_translation', 1)
            ->getAllIds();
    }

    /**
     * Get translatable attributes
     *
     * @return string[]
     */
    protected function getTranslatableAttributes(): array
    {
        $fieldList = $this->translatableFields;
        $fieldList[] = 'question_id';

        return $fieldList;
    }

    /**
     * @inheritDoc
     */
    public function getPreparedEntityData(int $entityId): array
    {
        return $this->getPreparedEntityDataByQuestion(
            $this->questionRepository->getById($entityId)
        );
    }

    /**
     * @inheritDoc
     */
    public function getPreparedEntityDataByStoreId(int $entityId, int $storeId, bool $isSkipAttributes = false): array
    {
        $preparedEntityData = $this->getPreparedEntityData($entityId);

        if ($isSkipAttributes) {
            foreach (self::SKIPPED_ATTRIBUTES as $attributeCode) {
                unset($preparedEntityData[$attributeCode]);
            }
        }

        return $preparedEntityData;
    }

    /**
     * Get prepared entity data by Amasty FAQ Question
     *
     * @param QuestionModel $question
     * @return array
     */
    public function getPreparedEntityDataByQuestion(QuestionModel $question): array
    {
        $preparedData = [];
        $attributes = $this->getTranslatableAttributes();

        foreach ($attributes as $attributeCode) {
            if (empty($question->getData($attributeCode))) {
                continue;
            }

            $preparedData[$attributeCode] = $question->getData($attributeCode);
        }

        return $preparedData;
    }

    /**
     * @inheritDoc
     */
    public function saveTranslatedData(array $entityData, array $storeIds): void
    {
        if ($storeIds === []) {
            return;
        }

        $entityId = $entityData['question_id'];
        /** @var QuestionModel $originalQuestion */
        $originalQuestion = $this->questionRepository->getById($entityId);
        $originalIdentifier = $originalQuestion->getUrlKey();
        $originalQuestion = $this->updateOriginalQuestion($originalQuestion, $storeIds, $originalIdentifier);
        $questionIdentifier = $entityData['url_key'];

        unset($entityData['url_key']);
        unset($entityData['question_id']);

        foreach ($storeIds as $storeId) {
            $questionCollection = $this->collectionFactory->create()
                ->addFieldToFilter('original_url_key', $originalIdentifier)
                ->addStoreFilter($storeId, false);

            if ($questionCollection->getSize() === 0) {
                $questionCollectionByIdentifier = $this->collectionFactory->create()
                    ->addFieldToFilter('url_key', $questionIdentifier)
                    ->addStoreFilter($storeId, false);

                if ($questionCollectionByIdentifier->getSize() > 0) {
                    $questionIdentifier = $this->resolveUniqueIdentifier($questionIdentifier, $storeId);
                }
            }

            $question = $questionCollection->getSize() === 0 ?
                $this->questionFactory->create()
                    ->addData($originalQuestion->getData())
                    ->unsetData('store_id')
                    ->setUrlKey($questionIdentifier)
                    ->setId(null):
                $questionCollection->getFirstItem();

            $question->addData($entityData)
                ->setStores([$storeId]);

            $this->questionRepository->save($question);
        }
    }

    /**
     * Update original question
     * @param QuestionModel $question
     * @param array $storeIds
     * @param string $originalIdentifier
     * @return QuestionModel
     * @throws LocalizedException
     */
    protected function updateOriginalQuestion(QuestionModel $question, array $storeIds, string $originalIdentifier): QuestionModel
    {
        $originalQuestionStoreIds = $question->getStores();

        if (!is_array($originalQuestionStoreIds)) {
            $originalQuestionStoreIds = [(int)$originalQuestionStoreIds];
        }

        if (in_array(Store::DEFAULT_STORE_ID, $originalQuestionStoreIds)) {
            $storeList = $this->storeManager->getStores();
            $originalQuestionStoreIds = [];

            foreach ($storeList as $store) {
                if ($store->getId() !== Store::DEFAULT_STORE_ID) {
                    $originalQuestionStoreIds[] = $store->getId();
                }
            }
        }

        $intersectedStores = array_intersect($storeIds, $originalQuestionStoreIds);

        if ($intersectedStores !== [] || in_array(Store::DEFAULT_STORE_ID, $originalQuestionStoreIds)) {
            $updatedStores = array_diff($originalQuestionStoreIds, $storeIds);

            $question->setStores($updatedStores !== [] ? $updatedStores : [$this->getDefaultStoreId()]);
            $question->setData('original_url_key', $originalIdentifier)
                ->setData('rewrites_update_force', true);
            $this->questionRepository->save($question);
        }

        return $question;
    }

    /**
     * Resolve URL key to avoid duplicates
     */
    protected function resolveUniqueIdentifier(
        string $identifier,
        int $storeId,
        int $index = 0
    ): string {
        $preparedIdentifier = $index === 0 ? $identifier : $identifier . '-' . $index;
        $index++;

        $collection = $this->collectionFactory->create();
        $collection->addFieldToFilter('url_key', $preparedIdentifier);
        $collection->addStoreFilter($storeId, false);

        return $collection->getSize() ?
            $this->resolveUniqueIdentifier($identifier, $storeId, $index) :
            $preparedIdentifier;
    }

    /**
     * Is Entity Updated
     * @param QuestionModel $entity
     * @return bool
     */
    public function isUpdated(QuestionModel $entity): bool
    {
        $originalData = $entity->getOrigData();
        $originalQuestion = $this->questionFactory->create();
        $originalQuestion->setData($originalData);

        $originalValues = $this->getPreparedEntityDataByQuestion($originalQuestion);
        $currentValues = $this->getPreparedEntityDataByQuestion($entity);

        return crc32(json_encode($originalValues)) !== crc32(json_encode($currentValues));
    }

    /**
     * @inheritDoc
     */
    public function getDefaultStoreId(): int
    {
        return $this->configHelper->getDefaultStoreId();
    }
}
